/*
 * Decompiled with CFR 0.152.
 */
package org.prebid.server.cookie;

import io.vertx.core.buffer.Buffer;
import io.vertx.core.http.Cookie;
import io.vertx.core.http.CookieSameSite;
import io.vertx.core.logging.Logger;
import io.vertx.core.logging.LoggerFactory;
import io.vertx.ext.web.RoutingContext;
import java.time.Duration;
import java.util.Base64;
import java.util.Collections;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.TreeMap;
import org.apache.commons.lang3.StringUtils;
import org.prebid.server.cookie.PrioritizedCoopSyncProvider;
import org.prebid.server.cookie.UidsCookie;
import org.prebid.server.cookie.model.UidWithExpiry;
import org.prebid.server.cookie.model.UidsCookieUpdateResult;
import org.prebid.server.cookie.proto.Uids;
import org.prebid.server.json.DecodeException;
import org.prebid.server.json.JacksonMapper;
import org.prebid.server.metric.Metrics;
import org.prebid.server.model.HttpRequestContext;
import org.prebid.server.util.HttpUtil;

public class UidsCookieService {
    private static final Logger logger = LoggerFactory.getLogger(UidsCookieService.class);
    private static final String COOKIE_NAME = "uids";
    private static final int MIN_COOKIE_SIZE_BYTES = 500;
    private final String optOutCookieName;
    private final String optOutCookieValue;
    private final String hostCookieFamily;
    private final String hostCookieName;
    private final String hostCookieDomain;
    private final long ttlSeconds;
    private final int maxCookieSizeBytes;
    private final PrioritizedCoopSyncProvider prioritizedCoopSyncProvider;
    private final Metrics metrics;
    private final JacksonMapper mapper;

    public UidsCookieService(String optOutCookieName, String optOutCookieValue, String hostCookieFamily, String hostCookieName, String hostCookieDomain, int ttlDays, int maxCookieSizeBytes, PrioritizedCoopSyncProvider prioritizedCoopSyncProvider, Metrics metrics, JacksonMapper mapper) {
        if (maxCookieSizeBytes != 0 && maxCookieSizeBytes < 500) {
            throw new IllegalArgumentException("Configured cookie size is less than allowed minimum size of 500");
        }
        this.optOutCookieName = optOutCookieName;
        this.optOutCookieValue = optOutCookieValue;
        this.hostCookieFamily = hostCookieFamily;
        this.hostCookieName = hostCookieName;
        this.hostCookieDomain = StringUtils.isNotBlank((CharSequence)hostCookieDomain) ? hostCookieDomain : null;
        this.ttlSeconds = Duration.ofDays(ttlDays).getSeconds();
        this.maxCookieSizeBytes = maxCookieSizeBytes;
        this.prioritizedCoopSyncProvider = Objects.requireNonNull(prioritizedCoopSyncProvider);
        this.metrics = Objects.requireNonNull(metrics);
        this.mapper = Objects.requireNonNull(mapper);
    }

    public UidsCookie parseFromRequest(RoutingContext routingContext) {
        return this.parseFromCookies(HttpUtil.cookiesAsMap(routingContext));
    }

    public UidsCookie parseFromRequest(HttpRequestContext httpRequest) {
        return this.parseFromCookies(HttpUtil.cookiesAsMap(httpRequest));
    }

    UidsCookie parseFromCookies(Map<String, String> cookies) {
        Map<String, UidWithExpiry> uidsMap;
        Boolean optout;
        Uids parsedUids = this.parseUids(cookies);
        Uids.UidsBuilder uidsBuilder = Uids.builder().uidsLegacy(Collections.emptyMap());
        if (this.isOptedOut(cookies)) {
            optout = true;
            uidsMap = Collections.emptyMap();
        } else {
            optout = parsedUids != null ? parsedUids.getOptout() : null;
            uidsMap = this.enrichAndSanitizeUids(parsedUids, cookies);
        }
        return new UidsCookie(uidsBuilder.uids(uidsMap).optout(optout).build(), this.mapper);
    }

    public Uids parseUids(Map<String, String> cookies) {
        if (cookies.containsKey(COOKIE_NAME)) {
            String cookieValue = cookies.get(COOKIE_NAME);
            try {
                return this.mapper.decodeValue(Buffer.buffer((byte[])Base64.getUrlDecoder().decode(cookieValue)), Uids.class);
            }
            catch (IllegalArgumentException | DecodeException e) {
                logger.debug((Object)"Could not decode or parse {0} cookie value {1}", (Throwable)e, new Object[]{COOKIE_NAME, cookieValue});
            }
        }
        return null;
    }

    public Cookie toCookie(UidsCookie uidsCookie) {
        return this.makeCookie(uidsCookie);
    }

    private int cookieBytesLength(UidsCookie uidsCookie) {
        return this.makeCookie(uidsCookie).encode().getBytes().length;
    }

    private Cookie makeCookie(UidsCookie uidsCookie) {
        return Cookie.cookie((String)COOKIE_NAME, (String)Base64.getUrlEncoder().encodeToString(uidsCookie.toJson().getBytes())).setPath("/").setSameSite(CookieSameSite.NONE).setSecure(true).setMaxAge(this.ttlSeconds).setDomain(this.hostCookieDomain);
    }

    public String parseHostCookie(Map<String, String> cookies) {
        return this.hostCookieName != null ? cookies.get(this.hostCookieName) : null;
    }

    public String parseHostCookie(HttpRequestContext httpRequest) {
        return this.parseHostCookie(HttpUtil.cookiesAsMap(httpRequest));
    }

    private boolean isOptedOut(Map<String, String> cookies) {
        if (StringUtils.isNotBlank((CharSequence)this.optOutCookieName) && StringUtils.isNotBlank((CharSequence)this.optOutCookieValue)) {
            String cookieValue = cookies.get(this.optOutCookieName);
            return cookieValue != null && Objects.equals(cookieValue, this.optOutCookieValue);
        }
        return false;
    }

    private Map<String, UidWithExpiry> enrichAndSanitizeUids(Uids uids, Map<String, String> cookies) {
        String hostCookie;
        Map<String, String> legacyUids;
        Map<String, UidWithExpiry> originalUidsMap = uids != null ? uids.getUids() : null;
        TreeMap<String, UidWithExpiry> workingUidsMap = new TreeMap<String, UidWithExpiry>(String.CASE_INSENSITIVE_ORDER);
        if (originalUidsMap != null) {
            workingUidsMap.putAll(originalUidsMap);
        }
        Map<String, String> map = legacyUids = uids != null ? uids.getUidsLegacy() : null;
        if (workingUidsMap.isEmpty() && legacyUids != null) {
            legacyUids.forEach((key, value) -> workingUidsMap.put((String)key, UidWithExpiry.expired(value)));
        }
        if ((hostCookie = this.parseHostCookie(cookies)) != null && UidsCookieService.hostCookieDiffers(hostCookie, (UidWithExpiry)workingUidsMap.get(this.hostCookieFamily))) {
            workingUidsMap.put(this.hostCookieFamily, UidWithExpiry.live(hostCookie));
        }
        workingUidsMap.entrySet().removeIf(UidsCookieService::facebookSentinelOrEmpty);
        return workingUidsMap;
    }

    private static boolean hostCookieDiffers(String hostCookie, UidWithExpiry uid) {
        return uid == null || !Objects.equals(hostCookie, uid.getUid());
    }

    private static boolean facebookSentinelOrEmpty(Map.Entry<String, UidWithExpiry> entry) {
        return UidsCookie.isFacebookSentinel(entry.getKey(), entry.getValue().getUid()) || StringUtils.isEmpty((CharSequence)entry.getValue().getUid());
    }

    public UidsCookieUpdateResult updateUidsCookie(UidsCookie uidsCookie, String familyName, String uid) {
        UidsCookie initialCookie = this.trimToLimit(UidsCookieService.removeExpiredUids(uidsCookie));
        if (StringUtils.isBlank((CharSequence)uid)) {
            return UidsCookieUpdateResult.unaltered(initialCookie.deleteUid(familyName));
        }
        if (UidsCookie.isFacebookSentinel(familyName, uid)) {
            return UidsCookieUpdateResult.unaltered(initialCookie);
        }
        return this.updateUidsCookieByPriority(initialCookie, familyName, uid);
    }

    private static UidsCookie removeExpiredUids(UidsCookie uidsCookie) {
        Set<String> families = uidsCookie.getCookieUids().getUids().keySet();
        UidsCookie updatedCookie = uidsCookie;
        for (String family : families) {
            updatedCookie = updatedCookie.hasLiveUidFrom(family) ? updatedCookie : updatedCookie.deleteUid(family);
        }
        return updatedCookie;
    }

    private UidsCookieUpdateResult updateUidsCookieByPriority(UidsCookie uidsCookie, String familyName, String uid) {
        UidsCookie updatedCookie = uidsCookie.updateUid(familyName, uid);
        if (!this.cookieExceededMaxLength(updatedCookie)) {
            return UidsCookieUpdateResult.updated(updatedCookie);
        }
        if (!this.prioritizedCoopSyncProvider.hasPrioritizedBidders() || this.prioritizedCoopSyncProvider.isPrioritizedFamily(familyName)) {
            return UidsCookieUpdateResult.updated(this.trimToLimit(updatedCookie));
        }
        this.metrics.updateUserSyncSizeBlockedMetric(familyName);
        return UidsCookieUpdateResult.unaltered(uidsCookie);
    }

    private boolean cookieExceededMaxLength(UidsCookie uidsCookie) {
        return this.maxCookieSizeBytes > 0 && this.cookieBytesLength(uidsCookie) > this.maxCookieSizeBytes;
    }

    private UidsCookie trimToLimit(UidsCookie uidsCookie) {
        if (!this.cookieExceededMaxLength(uidsCookie)) {
            return uidsCookie;
        }
        UidsCookie trimmedUids = uidsCookie;
        Iterator<String> familyToRemoveIterator = this.cookieFamilyNamesByAscendingPriority(uidsCookie);
        while (familyToRemoveIterator.hasNext() && this.cookieExceededMaxLength(trimmedUids)) {
            String familyToRemove = familyToRemoveIterator.next();
            this.metrics.updateUserSyncSizedOutMetric(familyToRemove);
            trimmedUids = trimmedUids.deleteUid(familyToRemove);
        }
        return trimmedUids;
    }

    private Iterator<String> cookieFamilyNamesByAscendingPriority(UidsCookie uidsCookie) {
        return uidsCookie.getCookieUids().getUids().entrySet().stream().sorted(this::compareCookieFamilyNames).map(Map.Entry::getKey).toList().iterator();
    }

    private int compareCookieFamilyNames(Map.Entry<String, UidWithExpiry> left, Map.Entry<String, UidWithExpiry> right) {
        boolean leftPrioritized = this.prioritizedCoopSyncProvider.isPrioritizedFamily(left.getKey());
        boolean rightPrioritized = this.prioritizedCoopSyncProvider.isPrioritizedFamily(right.getKey());
        if (leftPrioritized && rightPrioritized || !leftPrioritized && !rightPrioritized) {
            return left.getValue().getExpires().compareTo(right.getValue().getExpires());
        }
        if (leftPrioritized) {
            return 1;
        }
        return -1;
    }

    public String hostCookieUidToSync(RoutingContext routingContext, String cookieFamilyName) {
        if (!StringUtils.equals((CharSequence)cookieFamilyName, (CharSequence)this.hostCookieFamily)) {
            return null;
        }
        Map<String, String> cookies = HttpUtil.cookiesAsMap(routingContext);
        String hostCookieUid = this.parseHostCookie(cookies);
        if (hostCookieUid == null) {
            return null;
        }
        boolean inSync = Optional.ofNullable(this.parseUids(cookies)).map(Uids::getUids).map(uids -> (UidWithExpiry)uids.get(cookieFamilyName)).map(UidWithExpiry::getUid).filter(uid -> StringUtils.equals((CharSequence)hostCookieUid, (CharSequence)uid)).isPresent();
        return inSync ? null : hostCookieUid;
    }
}

