/*
 * Decompiled with CFR 0.152.
 */
package org.prebid.server.validation;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.TreeNode;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.iab.openrtb.request.Asset;
import com.iab.openrtb.request.Audio;
import com.iab.openrtb.request.Banner;
import com.iab.openrtb.request.BidRequest;
import com.iab.openrtb.request.DataObject;
import com.iab.openrtb.request.Device;
import com.iab.openrtb.request.Dooh;
import com.iab.openrtb.request.Eid;
import com.iab.openrtb.request.EventTracker;
import com.iab.openrtb.request.Format;
import com.iab.openrtb.request.ImageObject;
import com.iab.openrtb.request.Imp;
import com.iab.openrtb.request.Metric;
import com.iab.openrtb.request.Native;
import com.iab.openrtb.request.Pmp;
import com.iab.openrtb.request.Regs;
import com.iab.openrtb.request.Request;
import com.iab.openrtb.request.Site;
import com.iab.openrtb.request.TitleObject;
import com.iab.openrtb.request.Uid;
import com.iab.openrtb.request.User;
import com.iab.openrtb.request.Video;
import com.iab.openrtb.request.VideoObject;
import com.iab.openrtb.request.ntv.ContextSubType;
import com.iab.openrtb.request.ntv.ContextType;
import com.iab.openrtb.request.ntv.DataAssetType;
import com.iab.openrtb.request.ntv.EventTrackingMethod;
import com.iab.openrtb.request.ntv.EventType;
import com.iab.openrtb.request.ntv.PlacementType;
import com.iab.openrtb.request.ntv.Protocol;
import io.vertx.core.logging.LoggerFactory;
import java.io.IOException;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Stream;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.prebid.server.bidder.BidderCatalog;
import org.prebid.server.json.JacksonMapper;
import org.prebid.server.log.ConditionalLogger;
import org.prebid.server.model.HttpRequestContext;
import org.prebid.server.proto.openrtb.ext.request.ExtDevice;
import org.prebid.server.proto.openrtb.ext.request.ExtDeviceInt;
import org.prebid.server.proto.openrtb.ext.request.ExtDevicePrebid;
import org.prebid.server.proto.openrtb.ext.request.ExtGranularityRange;
import org.prebid.server.proto.openrtb.ext.request.ExtImpPrebid;
import org.prebid.server.proto.openrtb.ext.request.ExtMediaTypePriceGranularity;
import org.prebid.server.proto.openrtb.ext.request.ExtPriceGranularity;
import org.prebid.server.proto.openrtb.ext.request.ExtRequest;
import org.prebid.server.proto.openrtb.ext.request.ExtRequestBidAdjustmentFactors;
import org.prebid.server.proto.openrtb.ext.request.ExtRequestPrebid;
import org.prebid.server.proto.openrtb.ext.request.ExtRequestPrebidData;
import org.prebid.server.proto.openrtb.ext.request.ExtRequestPrebidDataEidPermissions;
import org.prebid.server.proto.openrtb.ext.request.ExtRequestPrebidSchain;
import org.prebid.server.proto.openrtb.ext.request.ExtRequestTargeting;
import org.prebid.server.proto.openrtb.ext.request.ExtSite;
import org.prebid.server.proto.openrtb.ext.request.ExtStoredAuctionResponse;
import org.prebid.server.proto.openrtb.ext.request.ExtStoredBidResponse;
import org.prebid.server.proto.openrtb.ext.request.ExtUser;
import org.prebid.server.proto.openrtb.ext.request.ExtUserPrebid;
import org.prebid.server.proto.openrtb.ext.request.ImpMediaType;
import org.prebid.server.proto.openrtb.ext.response.BidType;
import org.prebid.server.util.HttpUtil;
import org.prebid.server.util.StreamUtil;
import org.prebid.server.validation.BidderParamValidator;
import org.prebid.server.validation.ValidationException;
import org.prebid.server.validation.model.ValidationResult;

public class RequestValidator {
    private static final ConditionalLogger conditionalLogger = new ConditionalLogger(LoggerFactory.getLogger(RequestValidator.class));
    private static final String PREBID_EXT = "prebid";
    private static final String BIDDER_EXT = "bidder";
    private static final String ASTERISK = "*";
    private static final Locale LOCALE = Locale.US;
    private static final Integer NATIVE_EXCHANGE_SPECIFIC_LOWER_BOUND = 500;
    private static final String DOCUMENTATION = "https://iabtechlab.com/wp-content/uploads/2016/07/OpenRTB-Native-Ads-Specification-Final-1.2.pdf";
    private final BidderCatalog bidderCatalog;
    private final BidderParamValidator bidderParamValidator;
    private final JacksonMapper mapper;
    private final double logSamplingRate;

    public RequestValidator(BidderCatalog bidderCatalog, BidderParamValidator bidderParamValidator, JacksonMapper mapper, double logSamplingRate) {
        this.bidderCatalog = Objects.requireNonNull(bidderCatalog);
        this.bidderParamValidator = Objects.requireNonNull(bidderParamValidator);
        this.mapper = Objects.requireNonNull(mapper);
        this.logSamplingRate = logSamplingRate;
    }

    public ValidationResult validate(BidRequest bidRequest, HttpRequestContext httpRequestContext) {
        ArrayList<String> warnings = new ArrayList<String>();
        try {
            boolean isSite;
            if (StringUtils.isBlank((CharSequence)bidRequest.getId())) {
                throw new ValidationException("request missing required field: \"id\"");
            }
            if (bidRequest.getTmax() != null && bidRequest.getTmax() < 0L) {
                throw new ValidationException("request.tmax must be nonnegative. Got " + bidRequest.getTmax());
            }
            this.validateCur(bidRequest.getCur());
            ExtRequest extRequest = bidRequest.getExt();
            ExtRequestPrebid extRequestPrebid = extRequest != null ? extRequest.getPrebid() : null;
            Map aliases = Collections.emptyMap();
            if (extRequestPrebid != null) {
                ExtRequestTargeting targeting = extRequestPrebid.getTargeting();
                if (targeting != null) {
                    this.validateTargeting(targeting);
                }
                aliases = (Map)ObjectUtils.defaultIfNull(extRequestPrebid.getAliases(), Collections.emptyMap());
                this.validateAliases(aliases);
                this.validateAliasesGvlIds(extRequestPrebid, aliases);
                this.validateBidAdjustmentFactors(extRequestPrebid.getBidadjustmentfactors(), aliases);
                this.validateExtBidPrebidData(extRequestPrebid.getData(), aliases);
                this.validateSchains(extRequestPrebid.getSchains());
            }
            if (CollectionUtils.isEmpty(bidRequest.getImp())) {
                throw new ValidationException("request.imp must contain at least one element");
            }
            List<Imp> imps = bidRequest.getImp();
            ArrayList<String> errors = new ArrayList<String>();
            HashMap<String, Integer> uniqueImps = new HashMap<String, Integer>();
            for (int i = 0; i < imps.size(); ++i) {
                String impId = imps.get(i).getId();
                if (uniqueImps.get(impId) != null) {
                    errors.add("request.imp[%d].id and request.imp[%d].id are both \"%s\". Imp IDs must be unique.".formatted(uniqueImps.get(impId), i, impId));
                }
                uniqueImps.put(impId, i);
            }
            if (CollectionUtils.isNotEmpty(errors)) {
                throw new ValidationException(String.join((CharSequence)System.lineSeparator(), errors));
            }
            for (int index = 0; index < bidRequest.getImp().size(); ++index) {
                this.validateImp(bidRequest.getImp().get(index), aliases, index, warnings);
            }
            ArrayList channels = new ArrayList();
            Optional.ofNullable(bidRequest.getApp()).ifPresent(ignored -> channels.add("request.app"));
            Optional.ofNullable(bidRequest.getDooh()).ifPresent(ignored -> channels.add("request.dooh"));
            Optional.ofNullable(bidRequest.getSite()).ifPresent(ignored -> channels.add("request.site"));
            boolean isApp = bidRequest.getApp() != null;
            boolean isDooh = !isApp && bidRequest.getDooh() != null;
            boolean bl = isSite = !isApp && !isDooh && bidRequest.getSite() != null;
            if (channels.isEmpty()) {
                throw new ValidationException("One of request.site or request.app or request.dooh must be defined");
            }
            if (channels.size() > 1) {
                String logMessage = String.join((CharSequence)" and ", channels) + " are present. Referer: " + httpRequestContext.getHeaders().get(HttpUtil.REFERER_HEADER);
                conditionalLogger.warn(logMessage, this.logSamplingRate);
            }
            if (isDooh) {
                this.validateDooh(bidRequest.getDooh());
            }
            if (isSite) {
                this.validateSite(bidRequest.getSite());
            }
            this.validateDevice(bidRequest.getDevice());
            this.validateUser(bidRequest.getUser(), aliases);
            this.validateRegs(bidRequest.getRegs());
        }
        catch (ValidationException e) {
            return ValidationResult.error(e.getMessage());
        }
        return ValidationResult.success(warnings);
    }

    private void validateCur(List<String> currencies) throws ValidationException {
        if (CollectionUtils.isEmpty(currencies)) {
            throw new ValidationException("currency was not defined either in request.cur or in configuration field adServerCurrency");
        }
    }

    private void validateAliasesGvlIds(ExtRequestPrebid extRequestPrebid, Map<String, String> aliases) throws ValidationException {
        Map aliasGvlIds = MapUtils.emptyIfNull(extRequestPrebid.getAliasgvlids());
        for (Map.Entry aliasToGvlId : aliasGvlIds.entrySet()) {
            if (!aliases.containsKey(aliasToGvlId.getKey())) {
                throw new ValidationException("request.ext.prebid.aliasgvlids. vendorId %s refers to unknown bidder alias: %s", aliasToGvlId.getValue(), aliasToGvlId.getKey());
            }
            if ((Integer)aliasToGvlId.getValue() >= 1) continue;
            throw new ValidationException("request.ext.prebid.aliasgvlids. Invalid vendorId %s for alias: %s. Choose a different vendorId, or remove this entry.", aliasToGvlId.getValue(), aliasToGvlId.getKey());
        }
    }

    private void validateBidAdjustmentFactors(ExtRequestBidAdjustmentFactors adjustmentFactors, Map<String, String> aliases) throws ValidationException {
        EnumMap<ImpMediaType, Map<String, BigDecimal>> adjustmentsMediaTypeFactors;
        Map<Object, Object> bidderAdjustments = adjustmentFactors != null ? adjustmentFactors.getAdjustments() : Collections.emptyMap();
        for (Map.Entry<Object, Object> bidderAdjustment : bidderAdjustments.entrySet()) {
            String bidder = (String)bidderAdjustment.getKey();
            if (this.isUnknownBidderOrAlias(bidder, aliases)) {
                throw new ValidationException("request.ext.prebid.bidadjustmentfactors.%s is not a known bidder or alias", bidder);
            }
            BigDecimal adjustmentFactor = (BigDecimal)bidderAdjustment.getValue();
            if (adjustmentFactor.compareTo(BigDecimal.ZERO) > 0) continue;
            throw new ValidationException("request.ext.prebid.bidadjustmentfactors.%s must be a positive number. Got %s", bidder, RequestValidator.format(adjustmentFactor));
        }
        EnumMap<ImpMediaType, Map<String, BigDecimal>> enumMap = adjustmentsMediaTypeFactors = adjustmentFactors != null ? adjustmentFactors.getMediatypes() : null;
        if (adjustmentsMediaTypeFactors == null) {
            return;
        }
        for (Map.Entry entry : adjustmentsMediaTypeFactors.entrySet()) {
            this.validateBidAdjustmentFactorsByMediatype((ImpMediaType)((Object)entry.getKey()), (Map)entry.getValue(), aliases);
        }
    }

    private void validateBidAdjustmentFactorsByMediatype(ImpMediaType mediaType, Map<String, BigDecimal> bidderAdjustments, Map<String, String> aliases) throws ValidationException {
        for (Map.Entry<String, BigDecimal> bidderAdjustment : bidderAdjustments.entrySet()) {
            String bidder = bidderAdjustment.getKey();
            if (this.isUnknownBidderOrAlias(bidder, aliases)) {
                throw new ValidationException("request.ext.prebid.bidadjustmentfactors.%s.%s is not a known bidder or alias", new Object[]{mediaType, bidder});
            }
            BigDecimal adjustmentFactor = bidderAdjustment.getValue();
            if (adjustmentFactor.compareTo(BigDecimal.ZERO) > 0) continue;
            throw new ValidationException("request.ext.prebid.bidadjustmentfactors.%s.%s must be a positive number. Got %s", new Object[]{mediaType, bidder, RequestValidator.format(adjustmentFactor)});
        }
    }

    private void validateSchains(List<ExtRequestPrebidSchain> schains) throws ValidationException {
        if (schains == null) {
            return;
        }
        HashSet<String> schainBidders = new HashSet<String>();
        for (ExtRequestPrebidSchain schain : schains) {
            List<String> bidders;
            if (schain == null || (bidders = schain.getBidders()) == null) continue;
            for (String bidder : bidders) {
                if (schainBidders.contains(bidder)) {
                    throw new ValidationException("request.ext.prebid.schains contains multiple schains for bidder %s; it must contain no more than one per bidder.", bidder);
                }
                schainBidders.add(bidder);
            }
        }
    }

    private void validateExtBidPrebidData(ExtRequestPrebidData data, Map<String, String> aliases) throws ValidationException {
        if (data != null) {
            this.validateEidPermissions(data.getEidPermissions(), aliases);
        }
    }

    private void validateEidPermissions(List<ExtRequestPrebidDataEidPermissions> eidPermissions, Map<String, String> aliases) throws ValidationException {
        if (eidPermissions == null) {
            return;
        }
        for (ExtRequestPrebidDataEidPermissions eidPermission : eidPermissions) {
            if (eidPermission == null) {
                throw new ValidationException("request.ext.prebid.data.eidpermissions[i] can't be null");
            }
            this.validateEidPermissionSource(eidPermission.getSource());
            this.validateEidPermissionBidders(eidPermission.getBidders(), aliases);
        }
    }

    private void validateEidPermissionSource(String source) throws ValidationException {
        if (StringUtils.isEmpty((CharSequence)source)) {
            throw new ValidationException("Missing required value request.ext.prebid.data.eidPermissions[].source");
        }
    }

    private void validateEidPermissionBidders(List<String> bidders, Map<String, String> aliases) throws ValidationException {
        if (CollectionUtils.isEmpty(bidders)) {
            throw new ValidationException("request.ext.prebid.data.eidpermissions[].bidders[] required values but was empty or null");
        }
        for (String bidder : bidders) {
            if (this.bidderCatalog.isValidName(bidder) || this.bidderCatalog.isValidName(aliases.get(bidder)) || !ObjectUtils.notEqual((Object)bidder, (Object)ASTERISK)) continue;
            throw new ValidationException("request.ext.prebid.data.eidPermissions[].bidders[] unrecognized biddercode: '%s'", bidder);
        }
    }

    private boolean isUnknownBidderOrAlias(String bidder, Map<String, String> aliases) {
        return !this.bidderCatalog.isValidName(bidder) && !aliases.containsKey(bidder);
    }

    private static String format(BigDecimal value) {
        return String.format(LOCALE, "%f", value);
    }

    private void validateTargeting(ExtRequestTargeting extRequestTargeting) throws ValidationException {
        JsonNode pricegranularity = extRequestTargeting.getPricegranularity();
        if (pricegranularity != null && !pricegranularity.isNull()) {
            this.validateExtPriceGranularity(pricegranularity, null);
        }
        this.validateMediaTypePriceGranularity(extRequestTargeting.getMediatypepricegranularity());
        Boolean includeWinners = extRequestTargeting.getIncludewinners();
        Boolean includeBidderKeys = extRequestTargeting.getIncludebidderkeys();
        if (Objects.equals(includeWinners, false) && Objects.equals(includeBidderKeys, false)) {
            throw new ValidationException("ext.prebid.targeting: At least one of includewinners or includebidderkeys must be enabled to enable targeting support");
        }
    }

    private void validateExtPriceGranularity(JsonNode priceGranularity, BidType type) throws ValidationException {
        ExtPriceGranularity extPriceGranularity;
        try {
            extPriceGranularity = (ExtPriceGranularity)this.mapper.mapper().treeToValue((TreeNode)priceGranularity, ExtPriceGranularity.class);
        }
        catch (JsonProcessingException e) {
            throw new ValidationException("Error while parsing request.ext.prebid.targeting.%s".formatted(type == null ? "pricegranularity" : "mediatypepricegranularity." + type));
        }
        Integer precision = extPriceGranularity.getPrecision();
        if (precision != null && precision < 0) {
            throw new ValidationException("%srice granularity error: precision must be non-negative".formatted(type == null ? "P" : StringUtils.capitalize((String)type.name()) + " p"));
        }
        RequestValidator.validateGranularityRanges(extPriceGranularity.getRanges());
    }

    private void validateMediaTypePriceGranularity(ExtMediaTypePriceGranularity mediaTypePriceGranularity) throws ValidationException {
        if (mediaTypePriceGranularity != null) {
            boolean isNativeNull;
            ObjectNode banner = mediaTypePriceGranularity.getBanner();
            ObjectNode video = mediaTypePriceGranularity.getVideo();
            ObjectNode xNative = mediaTypePriceGranularity.getXNative();
            boolean isBannerNull = banner == null || banner.isNull();
            boolean isVideoNull = video == null || video.isNull();
            boolean bl = isNativeNull = xNative == null || xNative.isNull();
            if (isBannerNull && isVideoNull && isNativeNull) {
                throw new ValidationException("Media type price granularity error: must have at least one media type present");
            }
            if (!isBannerNull) {
                this.validateExtPriceGranularity((JsonNode)banner, BidType.banner);
            }
            if (!isVideoNull) {
                this.validateExtPriceGranularity((JsonNode)video, BidType.video);
            }
            if (!isNativeNull) {
                this.validateExtPriceGranularity((JsonNode)xNative, BidType.xNative);
            }
        }
    }

    private static void validateGranularityRanges(List<ExtGranularityRange> ranges) throws ValidationException {
        if (CollectionUtils.isEmpty(ranges)) {
            throw new ValidationException("Price granularity error: empty granularity definition supplied");
        }
        BigDecimal previousRangeMax = null;
        for (ExtGranularityRange range : ranges) {
            BigDecimal rangeMax = range.getMax();
            RequestValidator.validateGranularityRangeMax(rangeMax);
            RequestValidator.validateGranularityRangeIncrement(range);
            RequestValidator.validateGranularityRangeMaxOrdering(previousRangeMax, rangeMax);
            previousRangeMax = rangeMax;
        }
    }

    private static void validateGranularityRangeMax(BigDecimal rangeMax) throws ValidationException {
        if (rangeMax == null) {
            throw new ValidationException("Price granularity error: max value should not be missed");
        }
    }

    private static void validateGranularityRangeMaxOrdering(BigDecimal previousRangeMax, BigDecimal rangeMax) throws ValidationException {
        if (previousRangeMax != null && previousRangeMax.compareTo(rangeMax) > 0) {
            throw new ValidationException("Price granularity error: range list must be ordered with increasing \"max\"");
        }
    }

    private static void validateGranularityRangeIncrement(ExtGranularityRange range) throws ValidationException {
        BigDecimal increment = range.getIncrement();
        if (increment == null || increment.compareTo(BigDecimal.ZERO) <= 0) {
            throw new ValidationException("Price granularity error: increment must be a nonzero positive number");
        }
    }

    private void validateAliases(Map<String, String> aliases) throws ValidationException {
        for (Map.Entry<String, String> aliasToBidder : aliases.entrySet()) {
            String alias = aliasToBidder.getKey();
            String coreBidder = aliasToBidder.getValue();
            if (!this.bidderCatalog.isValidName(coreBidder)) {
                throw new ValidationException("request.ext.prebid.aliases.%s refers to unknown bidder: %s".formatted(alias, coreBidder));
            }
            if (!this.bidderCatalog.isActive(coreBidder)) {
                throw new ValidationException("request.ext.prebid.aliases.%s refers to disabled bidder: %s".formatted(alias, coreBidder));
            }
            if (!alias.equals(coreBidder)) continue;
            throw new ValidationException("request.ext.prebid.aliases.%s defines a no-op alias. Choose a different alias, or remove this entry".formatted(alias));
        }
    }

    private void validateSite(Site site) throws ValidationException {
        if (site != null) {
            Integer amp;
            if (StringUtils.isBlank((CharSequence)site.getId()) && StringUtils.isBlank((CharSequence)site.getPage())) {
                throw new ValidationException("request.site should include at least one of request.site.id or request.site.page");
            }
            ExtSite siteExt = site.getExt();
            if (siteExt != null && (amp = siteExt.getAmp()) != null && (amp < 0 || amp > 1)) {
                throw new ValidationException("request.site.ext.amp must be either 1, 0, or undefined");
            }
        }
    }

    private void validateDooh(Dooh dooh) throws ValidationException {
        if (dooh.getId() == null && CollectionUtils.isEmpty(dooh.getVenuetype())) {
            throw new ValidationException("request.dooh should include at least one of request.dooh.id or request.dooh.venuetype.");
        }
    }

    private void validateDevice(Device device) throws ValidationException {
        ExtDevice extDevice;
        ExtDevice extDevice2 = extDevice = device != null ? device.getExt() : null;
        if (extDevice != null) {
            ExtDeviceInt interstitial;
            ExtDevicePrebid extDevicePrebid = extDevice.getPrebid();
            ExtDeviceInt extDeviceInt = interstitial = extDevicePrebid != null ? extDevicePrebid.getInterstitial() : null;
            if (interstitial != null) {
                this.validateInterstitial(interstitial);
            }
        }
    }

    private void validateInterstitial(ExtDeviceInt interstitial) throws ValidationException {
        Integer minWidthPerc = interstitial.getMinWidthPerc();
        if (minWidthPerc == null || minWidthPerc < 0 || minWidthPerc > 100) {
            throw new ValidationException("request.device.ext.prebid.interstitial.minwidthperc must be a number between 0 and 100");
        }
        Integer minHeightPerc = interstitial.getMinHeightPerc();
        if (minHeightPerc == null || minHeightPerc < 0 || minHeightPerc > 100) {
            throw new ValidationException("request.device.ext.prebid.interstitial.minheightperc must be a number between 0 and 100");
        }
    }

    private void validateUser(User user, Map<String, String> aliases) throws ValidationException {
        if (user != null) {
            this.validateUserExt(user.getExt(), aliases);
            List<Eid> eids = user.getEids();
            if (eids != null) {
                for (int index = 0; index < eids.size(); ++index) {
                    Eid eid = eids.get(index);
                    if (StringUtils.isBlank((CharSequence)eid.getSource())) {
                        throw new ValidationException("request.user.eids[%d] missing required field: \"source\"", index);
                    }
                    List<Uid> eidUids = eid.getUids();
                    if (CollectionUtils.isEmpty(eidUids)) {
                        throw new ValidationException("request.user.eids[%d].uids must contain at least one element", index);
                    }
                    for (int uidsIndex = 0; uidsIndex < eidUids.size(); ++uidsIndex) {
                        Uid uid = eidUids.get(uidsIndex);
                        if (!StringUtils.isBlank((CharSequence)uid.getId())) continue;
                        throw new ValidationException("request.user.eids[%d].uids[%d] missing required field: \"id\"", index, uidsIndex);
                    }
                }
            }
        }
    }

    private void validateUserExt(ExtUser extUser, Map<String, String> aliases) throws ValidationException {
        ExtUserPrebid prebid;
        if (extUser != null && (prebid = extUser.getPrebid()) != null) {
            Map<String, String> buyerUids = prebid.getBuyeruids();
            if (MapUtils.isEmpty(buyerUids)) {
                throw new ValidationException("request.user.ext.prebid requires a \"buyeruids\" property with at least one ID defined. If none exist, then request.user.ext.prebid should not be defined");
            }
            for (String bidder : buyerUids.keySet()) {
                if (!this.isUnknownBidderOrAlias(bidder, aliases)) continue;
                throw new ValidationException("request.user.ext.%s is neither a known bidder name nor an alias in request.ext.prebid.aliases", bidder);
            }
        }
    }

    private void validateRegs(Regs regs) throws ValidationException {
        Integer gdpr;
        Integer n = gdpr = regs != null ? regs.getGdpr() : null;
        if (gdpr != null && gdpr != 0 && gdpr != 1) {
            throw new ValidationException("request.regs.ext.gdpr must be either 0 or 1");
        }
    }

    private void validateImp(Imp imp, Map<String, String> aliases, int index, List<String> warnings) throws ValidationException {
        if (StringUtils.isBlank((CharSequence)imp.getId())) {
            throw new ValidationException("request.imp[%d] missing required field: \"id\"", index);
        }
        if (imp.getMetric() != null && !imp.getMetric().isEmpty()) {
            this.validateMetrics(imp.getMetric(), index);
        }
        if (imp.getBanner() == null && imp.getVideo() == null && imp.getAudio() == null && imp.getXNative() == null) {
            throw new ValidationException("request.imp[%d] must contain at least one of \"banner\", \"video\", \"audio\", or \"native\"", index);
        }
        boolean isInterstitialImp = Objects.equals(imp.getInstl(), 1);
        this.validateBanner(imp.getBanner(), isInterstitialImp, index);
        this.validateVideoMimes(imp.getVideo(), index);
        this.validateAudioMimes(imp.getAudio(), index);
        this.fillAndValidateNative(imp.getXNative(), index);
        this.validatePmp(imp.getPmp(), index);
        this.validateImpExt(imp.getExt(), aliases, index, warnings);
    }

    private void fillAndValidateNative(Native xNative, int impIndex) throws ValidationException {
        if (xNative == null) {
            return;
        }
        Request nativeRequest = this.parseNativeRequest(xNative.getRequest(), impIndex);
        this.validateNativeContextTypes(nativeRequest.getContext(), nativeRequest.getContextsubtype(), impIndex);
        this.validateNativePlacementType(nativeRequest.getPlcmttype(), impIndex);
        List<Asset> updatedAssets = this.validateAndGetUpdatedNativeAssets(nativeRequest.getAssets(), impIndex);
        this.validateNativeEventTrackers(nativeRequest.getEventtrackers(), impIndex);
        xNative.setRequest(this.toEncodedRequest(nativeRequest, updatedAssets));
    }

    private Request parseNativeRequest(String rawStringNativeRequest, int impIndex) throws ValidationException {
        if (StringUtils.isBlank((CharSequence)rawStringNativeRequest)) {
            throw new ValidationException("request.imp[%d].native contains empty request value", impIndex);
        }
        try {
            return (Request)this.mapper.mapper().readValue(rawStringNativeRequest, Request.class);
        }
        catch (IOException e) {
            throw new ValidationException("Error while parsing request.imp[%d].native.request: %s", impIndex, ExceptionUtils.getMessage((Throwable)e));
        }
    }

    private void validateNativeContextTypes(Integer context, Integer contextSubType, int index) throws ValidationException {
        int subType;
        int type;
        int n = type = context != null ? context : 0;
        if (type == 0) {
            return;
        }
        if (type < ContextType.CONTENT.getValue() || type > ContextType.PRODUCT.getValue() && type < NATIVE_EXCHANGE_SPECIFIC_LOWER_BOUND) {
            throw new ValidationException("request.imp[%d].native.request.context is invalid. See " + RequestValidator.documentationOnPage(39), index);
        }
        int n2 = subType = contextSubType != null ? contextSubType : 0;
        if (subType < 0) {
            throw new ValidationException("request.imp[%d].native.request.contextsubtype is invalid. See " + RequestValidator.documentationOnPage(39), index);
        }
        if (subType == 0) {
            return;
        }
        if (subType >= ContextSubType.GENERAL.getValue() && subType <= ContextSubType.USER_GENERATED.getValue()) {
            if (type != ContextType.CONTENT.getValue() && type < NATIVE_EXCHANGE_SPECIFIC_LOWER_BOUND) {
                throw new ValidationException("request.imp[%d].native.request.context is %d, but contextsubtype is %d. This is an invalid combination. See " + RequestValidator.documentationOnPage(39), index, context, contextSubType);
            }
            return;
        }
        if (subType >= ContextSubType.SOCIAL.getValue() && subType <= ContextSubType.CHAT.getValue()) {
            if (type != ContextType.SOCIAL.getValue() && type < NATIVE_EXCHANGE_SPECIFIC_LOWER_BOUND) {
                throw new ValidationException("request.imp[%d].native.request.context is %d, but contextsubtype is %d. This is an invalid combination. See " + RequestValidator.documentationOnPage(39), index, context, contextSubType);
            }
            return;
        }
        if (subType >= ContextSubType.SELLING.getValue() && subType <= ContextSubType.PRODUCT_REVIEW.getValue()) {
            if (type != ContextType.PRODUCT.getValue() && type < NATIVE_EXCHANGE_SPECIFIC_LOWER_BOUND) {
                throw new ValidationException("request.imp[%d].native.request.context is %d, but contextsubtype is %d. This is an invalid combination. See " + RequestValidator.documentationOnPage(39), index, context, contextSubType);
            }
            return;
        }
        if (subType < NATIVE_EXCHANGE_SPECIFIC_LOWER_BOUND) {
            throw new ValidationException("request.imp[%d].native.request.contextsubtype is invalid. See " + RequestValidator.documentationOnPage(39), index);
        }
    }

    private void validateNativePlacementType(Integer placementType, int index) throws ValidationException {
        int type;
        int n = type = placementType != null ? placementType : 0;
        if (type == 0) {
            return;
        }
        if (type < PlacementType.FEED.getValue() || type > PlacementType.RECOMMENDATION_WIDGET.getValue() && type < NATIVE_EXCHANGE_SPECIFIC_LOWER_BOUND) {
            throw new ValidationException("request.imp[%d].native.request.plcmttype is invalid. See " + RequestValidator.documentationOnPage(40), index, type);
        }
    }

    private List<Asset> validateAndGetUpdatedNativeAssets(List<Asset> assets, int impIndex) throws ValidationException {
        if (CollectionUtils.isEmpty(assets)) {
            throw new ValidationException("request.imp[%d].native.request.assets must be an array containing at least one object", impIndex);
        }
        ArrayList<Asset> updatedAssets = new ArrayList<Asset>();
        for (int i = 0; i < assets.size(); ++i) {
            Asset asset = assets.get(i);
            this.validateNativeAsset(asset, impIndex, i);
            Asset updatedAsset = asset.getId() != null ? asset : asset.toBuilder().id(i).build();
            boolean hasAssetWithId = updatedAssets.stream().map(Asset::getId).anyMatch(id -> id.equals(updatedAsset.getId()));
            if (hasAssetWithId) {
                throw new ValidationException("request.imp[%d].native.request.assets[%d].id is already being used by another asset. Each asset ID must be unique.", impIndex, i);
            }
            updatedAssets.add(updatedAsset);
        }
        return updatedAssets;
    }

    private void validateNativeAsset(Asset asset, int impIndex, int assetIndex) throws ValidationException {
        TitleObject title = asset.getTitle();
        ImageObject image = asset.getImg();
        VideoObject video = asset.getVideo();
        DataObject data = asset.getData();
        long assetsCount = Stream.of(title, image, video, data).filter(Objects::nonNull).count();
        if (assetsCount > 1L) {
            throw new ValidationException("request.imp[%d].native.request.assets[%d] must define at most one of {title, img, video, data}", impIndex, assetIndex);
        }
        this.validateNativeAssetTitle(title, impIndex, assetIndex);
        this.validateNativeAssetVideo(video, impIndex, assetIndex);
        this.validateNativeAssetData(data, impIndex, assetIndex);
    }

    private void validateNativeAssetTitle(TitleObject title, int impIndex, int assetIndex) throws ValidationException {
        if (title != null && (title.getLen() == null || title.getLen() < 1)) {
            throw new ValidationException("request.imp[%d].native.request.assets[%d].title.len must be a positive integer", impIndex, assetIndex);
        }
    }

    private void validateNativeAssetData(DataObject data, int impIndex, int assetIndex) throws ValidationException {
        if (data == null || data.getType() == null) {
            return;
        }
        Integer type = data.getType();
        if (type < DataAssetType.SPONSORED.getValue() || type > DataAssetType.CTA_TEXT.getValue() && type < NATIVE_EXCHANGE_SPECIFIC_LOWER_BOUND) {
            throw new ValidationException("request.imp[%d].native.request.assets[%d].data.type is invalid. See section 7.4: " + RequestValidator.documentationOnPage(40), impIndex, assetIndex);
        }
    }

    private void validateNativeAssetVideo(VideoObject video, int impIndex, int assetIndex) throws ValidationException {
        if (video == null) {
            return;
        }
        if (CollectionUtils.isEmpty(video.getMimes())) {
            throw new ValidationException("request.imp[%d].native.request.assets[%d].video.mimes must be an array with at least one MIME type", impIndex, assetIndex);
        }
        if (video.getMinduration() == null || video.getMinduration() < 1) {
            throw new ValidationException("request.imp[%d].native.request.assets[%d].video.minduration must be a positive integer", impIndex, assetIndex);
        }
        if (video.getMaxduration() == null || video.getMaxduration() < 1) {
            throw new ValidationException("request.imp[%d].native.request.assets[%d].video.maxduration must be a positive integer", impIndex, assetIndex);
        }
        this.validateNativeVideoProtocols(video.getProtocols(), impIndex, assetIndex);
    }

    private void validateNativeVideoProtocols(List<Integer> protocols, int impIndex, int assetIndex) throws ValidationException {
        if (CollectionUtils.isEmpty(protocols)) {
            throw new ValidationException("request.imp[%d].native.request.assets[%d].video.protocols must be an array with at least one element", impIndex, assetIndex);
        }
        for (int i = 0; i < protocols.size(); ++i) {
            this.validateNativeVideoProtocol(protocols.get(i), impIndex, assetIndex, i);
        }
    }

    private void validateNativeVideoProtocol(Integer protocol, int impIndex, int assetIndex, int protocolIndex) throws ValidationException {
        if (protocol < Protocol.VAST10.getValue() || protocol > Protocol.DAAST10_WRAPPER.getValue()) {
            throw new ValidationException("request.imp[%d].native.request.assets[%d].video.protocols[%d] must be in the range [1, 10]. Got %d", impIndex, assetIndex, protocolIndex, protocol);
        }
    }

    private void validateNativeEventTrackers(List<EventTracker> eventTrackers, int impIndex) throws ValidationException {
        if (CollectionUtils.isNotEmpty(eventTrackers)) {
            for (int eventTrackerIndex = 0; eventTrackerIndex < eventTrackers.size(); ++eventTrackerIndex) {
                this.validateNativeEventTracker(eventTrackers.get(eventTrackerIndex), impIndex, eventTrackerIndex);
            }
        }
    }

    private void validateNativeEventTracker(EventTracker eventTracker, int impIndex, int eventIndex) throws ValidationException {
        if (eventTracker != null) {
            int event;
            int n = event = eventTracker.getEvent() != null ? eventTracker.getEvent() : 0;
            if (event != 0 && (event < EventType.IMPRESSION.getValue() || event > EventType.VIEWABLE_VIDEO50.getValue() && event < NATIVE_EXCHANGE_SPECIFIC_LOWER_BOUND)) {
                throw new ValidationException("request.imp[%d].native.request.eventtrackers[%d].event is invalid. See section 7.6: " + RequestValidator.documentationOnPage(43), impIndex, eventIndex);
            }
            List<Integer> methods = eventTracker.getMethods();
            if (CollectionUtils.isEmpty(methods)) {
                throw new ValidationException("request.imp[%d].native.request.eventtrackers[%d].method is required. See section 7.7: " + RequestValidator.documentationOnPage(43), impIndex, eventIndex);
            }
            for (int methodIndex = 0; methodIndex < methods.size(); ++methodIndex) {
                int method;
                int n2 = method = methods.get(methodIndex) != null ? methods.get(methodIndex) : 0;
                if (method >= EventTrackingMethod.IMAGE.getValue() && (method <= EventTrackingMethod.JS.getValue() || event >= NATIVE_EXCHANGE_SPECIFIC_LOWER_BOUND)) continue;
                throw new ValidationException("request.imp[%d].native.request.eventtrackers[%d].methods[%d] is invalid. See section 7.7: " + RequestValidator.documentationOnPage(43), impIndex, eventIndex, methodIndex);
            }
        }
    }

    private static String documentationOnPage(int page) {
        return "%s#page=%d".formatted(DOCUMENTATION, page);
    }

    private String toEncodedRequest(Request nativeRequest, List<Asset> updatedAssets) {
        try {
            return this.mapper.mapper().writeValueAsString((Object)nativeRequest.toBuilder().assets(updatedAssets).build());
        }
        catch (JsonProcessingException e) {
            throw new IllegalArgumentException("Error while marshaling native request to the string", e);
        }
    }

    private void validateImpExt(ObjectNode ext, Map<String, String> aliases, int impIndex, List<String> warnings) throws ValidationException {
        this.validateImpExtPrebid(ext != null ? ext.get(PREBID_EXT) : null, aliases, impIndex, warnings);
    }

    private void validateImpExtPrebid(JsonNode extPrebidNode, Map<String, String> aliases, int impIndex, List<String> warnings) throws ValidationException {
        if (extPrebidNode == null) {
            throw new ValidationException("request.imp[%d].ext.prebid must be defined", impIndex);
        }
        if (!extPrebidNode.isObject()) {
            throw new ValidationException("request.imp[%d].ext.prebid must an object type", impIndex);
        }
        JsonNode extPrebidBidderNode = extPrebidNode.get(BIDDER_EXT);
        if (extPrebidBidderNode != null && !extPrebidBidderNode.isObject()) {
            throw new ValidationException("request.imp[%d].ext.prebid.bidder must be an object type", impIndex);
        }
        ExtImpPrebid extPrebid = this.parseExtImpPrebid((ObjectNode)extPrebidNode, impIndex);
        this.validateImpExtPrebidBidder(extPrebidBidderNode, extPrebid.getStoredAuctionResponse(), aliases, impIndex, warnings);
        this.validateImpExtPrebidStoredResponses(extPrebid, aliases, impIndex);
    }

    private void validateImpExtPrebidBidder(JsonNode extPrebidBidder, ExtStoredAuctionResponse storedAuctionResponse, Map<String, String> aliases, int impIndex, List<String> warnings) throws ValidationException {
        if (extPrebidBidder == null) {
            if (storedAuctionResponse != null) {
                return;
            }
            throw new ValidationException("request.imp[%d].ext.prebid.bidder must be defined", impIndex);
        }
        Iterator bidderExtensions = extPrebidBidder.fields();
        while (bidderExtensions.hasNext()) {
            Map.Entry bidderExtension = (Map.Entry)bidderExtensions.next();
            String bidder = (String)bidderExtension.getKey();
            try {
                this.validateImpBidderExtName(impIndex, bidderExtension, aliases.getOrDefault(bidder, bidder));
            }
            catch (ValidationException ex) {
                bidderExtensions.remove();
                warnings.add("WARNING: request.imp[%d].ext.prebid.bidder.%s was dropped with a reason: %s".formatted(impIndex, bidder, ex.getMessage()));
            }
        }
        if (extPrebidBidder.size() == 0) {
            warnings.add("WARNING: request.imp[%d].ext must contain at least one valid bidder".formatted(impIndex));
        }
    }

    private void validateImpExtPrebidStoredResponses(ExtImpPrebid extPrebid, Map<String, String> aliases, int impIndex) throws ValidationException {
        ExtStoredAuctionResponse extStoredAuctionResponse = extPrebid.getStoredAuctionResponse();
        if (extStoredAuctionResponse != null && extStoredAuctionResponse.getId() == null) {
            throw new ValidationException("request.imp[%d].ext.prebid.storedauctionresponse.id should be defined", impIndex);
        }
        List<ExtStoredBidResponse> storedBidResponses = extPrebid.getStoredBidResponse();
        if (CollectionUtils.isNotEmpty(storedBidResponses)) {
            ObjectNode bidderNode = extPrebid.getBidder();
            if (bidderNode == null || bidderNode.isEmpty()) {
                throw new ValidationException("request.imp[%d].ext.prebid.bidder should be defined for storedbidresponse".formatted(impIndex));
            }
            for (ExtStoredBidResponse storedBidResponse : storedBidResponses) {
                this.validateStoredBidResponse(storedBidResponse, bidderNode, aliases, impIndex);
            }
        }
    }

    private void validateStoredBidResponse(ExtStoredBidResponse extStoredBidResponse, ObjectNode bidderNode, Map<String, String> aliases, int impIndex) throws ValidationException {
        String bidder = extStoredBidResponse.getBidder();
        String id = extStoredBidResponse.getId();
        if (StringUtils.isEmpty((CharSequence)bidder)) {
            throw new ValidationException("request.imp[%d].ext.prebid.storedbidresponse.bidder was not defined".formatted(impIndex));
        }
        if (StringUtils.isEmpty((CharSequence)id)) {
            throw new ValidationException("Id was not defined for request.imp[%d].ext.prebid.storedbidresponse.id".formatted(impIndex));
        }
        String resolvedBidder = aliases.getOrDefault(bidder, bidder);
        if (!this.bidderCatalog.isValidName(resolvedBidder)) {
            throw new ValidationException("request.imp[%d].ext.prebid.storedbidresponse.bidder is not valid bidder".formatted(impIndex));
        }
        boolean noCorrespondentBidderParameters = StreamUtil.asStream(bidderNode.fieldNames()).noneMatch(impBidder -> impBidder.equals(resolvedBidder) || impBidder.equals(bidder));
        if (noCorrespondentBidderParameters) {
            throw new ValidationException("request.imp[%d].ext.prebid.storedbidresponse.bidder does not have correspondent bidder parameters".formatted(impIndex));
        }
    }

    private ExtImpPrebid parseExtImpPrebid(ObjectNode extImpPrebid, int impIndex) throws ValidationException {
        try {
            return (ExtImpPrebid)this.mapper.mapper().treeToValue((TreeNode)extImpPrebid, ExtImpPrebid.class);
        }
        catch (JsonProcessingException e) {
            throw new ValidationException(" bidRequest.imp[%d].ext.prebid: %s has invalid format".formatted(impIndex, e.getMessage()));
        }
    }

    private void validateImpBidderExtName(int impIndex, Map.Entry<String, JsonNode> bidderExtension, String bidderName) throws ValidationException {
        if (this.bidderCatalog.isValidName(bidderName)) {
            Set<String> messages = this.bidderParamValidator.validate(bidderName, bidderExtension.getValue());
            if (!messages.isEmpty()) {
                throw new ValidationException("request.imp[%d].ext.prebid.bidder.%s failed validation.\n%s", impIndex, bidderName, String.join((CharSequence)"\n", messages));
            }
        } else if (!this.bidderCatalog.isDeprecatedName(bidderName)) {
            throw new ValidationException("request.imp[%d].ext.prebid.bidder contains unknown bidder: %s", impIndex, bidderName);
        }
    }

    private void validatePmp(Pmp pmp, int impIndex) throws ValidationException {
        if (pmp != null && pmp.getDeals() != null) {
            for (int dealIndex = 0; dealIndex < pmp.getDeals().size(); ++dealIndex) {
                if (!StringUtils.isBlank((CharSequence)pmp.getDeals().get(dealIndex).getId())) continue;
                throw new ValidationException("request.imp[%d].pmp.deals[%d] missing required field: \"id\"", impIndex, dealIndex);
            }
        }
    }

    private void validateBanner(Banner banner, boolean isInterstitial, int impIndex) throws ValidationException {
        if (banner != null) {
            Integer width = banner.getW();
            Integer height = banner.getH();
            boolean hasWidth = RequestValidator.hasPositiveValue(width);
            boolean hasHeight = RequestValidator.hasPositiveValue(height);
            boolean hasSize = hasWidth && hasHeight;
            List<Format> format = banner.getFormat();
            if (CollectionUtils.isEmpty(format) && !hasSize && !isInterstitial) {
                throw new ValidationException("request.imp[%d].banner has no sizes. Define \"w\" and \"h\", or include \"format\" elements", impIndex);
            }
            if (width != null && height != null && !hasSize && !isInterstitial) {
                throw new ValidationException("Request imp[%d].banner must define a valid \"h\" and \"w\" properties", impIndex);
            }
            if (format != null) {
                for (int formatIndex = 0; formatIndex < format.size(); ++formatIndex) {
                    this.validateFormat(format.get(formatIndex), impIndex, formatIndex);
                }
            }
        }
    }

    private void validateFormat(Format format, int impIndex, int formatIndex) throws ValidationException {
        boolean usesRatios;
        boolean usesH = RequestValidator.hasPositiveValue(format.getH());
        boolean usesW = RequestValidator.hasPositiveValue(format.getW());
        boolean usesWmin = RequestValidator.hasPositiveValue(format.getWmin());
        boolean usesWratio = RequestValidator.hasPositiveValue(format.getWratio());
        boolean usesHratio = RequestValidator.hasPositiveValue(format.getHratio());
        boolean usesHW = usesH || usesW;
        boolean bl = usesRatios = usesWmin || usesWratio || usesHratio;
        if (usesHW && usesRatios) {
            throw new ValidationException("Request imp[%d].banner.format[%d] should define *either* {w, h} *or* {wmin, wratio, hratio}, but not both. If both are valid, send two \"format\" objects in the request", impIndex, formatIndex);
        }
        if (!usesHW && !usesRatios) {
            throw new ValidationException("Request imp[%d].banner.format[%d] should define *either* {w, h} (for static size requirements) *or* {wmin, wratio, hratio} (for flexible sizes) to be non-zero positive", impIndex, formatIndex);
        }
        if (!(!usesHW || usesH && usesW)) {
            throw new ValidationException("Request imp[%d].banner.format[%d] must define a valid \"h\" and \"w\" properties", impIndex, formatIndex);
        }
        if (!(!usesRatios || usesWmin && usesWratio && usesHratio)) {
            throw new ValidationException("Request imp[%d].banner.format[%d] must define a valid \"wmin\", \"wratio\", and \"hratio\" properties", impIndex, formatIndex);
        }
    }

    private void validateVideoMimes(Video video, int impIndex) throws ValidationException {
        if (video != null) {
            this.validateMimes(video.getMimes(), "request.imp[%d].video.mimes must contain at least one supported MIME type", impIndex);
        }
    }

    private void validateAudioMimes(Audio audio, int impIndex) throws ValidationException {
        if (audio != null) {
            this.validateMimes(audio.getMimes(), "request.imp[%d].audio.mimes must contain at least one supported MIME type", impIndex);
        }
    }

    private void validateMimes(List<String> mimes, String msg, int index) throws ValidationException {
        if (CollectionUtils.isEmpty(mimes)) {
            throw new ValidationException(msg, index);
        }
    }

    private void validateMetrics(List<Metric> metrics, int impIndex) throws ValidationException {
        for (int i = 0; i < metrics.size(); ++i) {
            Metric metric = metrics.get(i);
            if (StringUtils.isEmpty((CharSequence)metric.getType())) {
                throw new ValidationException("Missing request.imp[%d].metric[%d].type", impIndex, i);
            }
            Float value = metric.getValue();
            if (value != null && !((double)value.floatValue() < 0.0) && !((double)value.floatValue() > 1.0)) continue;
            throw new ValidationException("request.imp[%d].metric[%d].value must be in the range [0.0, 1.0]", impIndex, i);
        }
    }

    private static boolean hasPositiveValue(Integer value) {
        return value != null && value > 0;
    }
}

