/*
 * Decompiled with CFR 0.152.
 */
package org.prebid.server.privacy.gdpr.vendorlist;

import com.github.benmanes.caffeine.cache.Caffeine;
import io.netty.handler.codec.http.HttpResponseStatus;
import io.vertx.core.Future;
import io.vertx.core.Promise;
import io.vertx.core.Vertx;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.file.FileProps;
import io.vertx.core.file.FileSystem;
import io.vertx.core.file.FileSystemException;
import io.vertx.core.logging.Logger;
import io.vertx.core.logging.LoggerFactory;
import java.beans.ConstructorProperties;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Collection;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.stream.Collectors;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.prebid.server.exception.PreBidException;
import org.prebid.server.json.JacksonMapper;
import org.prebid.server.metric.Metrics;
import org.prebid.server.privacy.gdpr.vendorlist.proto.Vendor;
import org.prebid.server.privacy.gdpr.vendorlist.proto.VendorList;
import org.prebid.server.vertx.http.HttpClient;
import org.prebid.server.vertx.http.model.HttpClientResponse;

public class VendorListService {
    private static final Logger logger = LoggerFactory.getLogger(VendorListService.class);
    private static final int TCF_VERSION = 2;
    private static final String JSON_SUFFIX = ".json";
    private static final String VERSION_PLACEHOLDER = "{VERSION}";
    private final String cacheDir;
    private final String endpointTemplate;
    private final int defaultTimeoutMs;
    private final long refreshMissingListPeriodMs;
    private final boolean deprecated;
    private final Vertx vertx;
    private final FileSystem fileSystem;
    private final HttpClient httpClient;
    private final Metrics metrics;
    private final String generationVersion;
    protected final JacksonMapper mapper;
    private final Map<Integer, Map<Integer, Vendor>> cache;
    private final Map<Integer, Vendor> fallbackVendorList;
    private final Set<Integer> versionsToFallback;

    public VendorListService(String cacheDir, String endpointTemplate, int defaultTimeoutMs, long refreshMissingListPeriodMs, boolean deprecated, String fallbackVendorListPath, Vertx vertx, FileSystem fileSystem, HttpClient httpClient, Metrics metrics, String generationVersion, JacksonMapper mapper) {
        this.cacheDir = Objects.requireNonNull(cacheDir);
        this.endpointTemplate = Objects.requireNonNull(endpointTemplate);
        this.defaultTimeoutMs = defaultTimeoutMs;
        this.refreshMissingListPeriodMs = refreshMissingListPeriodMs;
        this.deprecated = deprecated;
        this.generationVersion = generationVersion;
        this.vertx = Objects.requireNonNull(vertx);
        this.fileSystem = Objects.requireNonNull(fileSystem);
        this.httpClient = Objects.requireNonNull(httpClient);
        this.metrics = Objects.requireNonNull(metrics);
        this.mapper = Objects.requireNonNull(mapper);
        VendorListService.createAndCheckWritePermissionsFor(fileSystem, cacheDir);
        this.cache = Objects.requireNonNull(this.createCache(fileSystem, cacheDir));
        Map<Integer, Vendor> map = this.fallbackVendorList = StringUtils.isNotBlank((CharSequence)fallbackVendorListPath) ? this.readFallbackVendorList(fallbackVendorListPath) : null;
        if (deprecated) {
            this.validateFallbackVendorListIfDeprecatedVersion();
        }
        this.versionsToFallback = this.fallbackVendorList != null ? ConcurrentHashMap.newKeySet() : null;
    }

    private void validateFallbackVendorListIfDeprecatedVersion() {
        if (Objects.isNull(this.fallbackVendorList)) {
            throw new PreBidException("No fallback vendorList for deprecated version present");
        }
    }

    public Future<Map<Integer, Vendor>> forVersion(int version) {
        if (version <= 0) {
            return Future.failedFuture((String)"TCF %d vendor list for version %s.%d not valid.".formatted(this.getTcfVersion(), this.generationVersion, version));
        }
        Map<Integer, Vendor> idToVendor = this.cache.get(version);
        if (idToVendor != null) {
            return Future.succeededFuture(idToVendor);
        }
        int tcf = this.getTcfVersion();
        if (this.shouldFallback(version)) {
            this.metrics.updatePrivacyTcfVendorListFallbackMetric(tcf);
            return Future.succeededFuture(this.fallbackVendorList);
        }
        this.metrics.updatePrivacyTcfVendorListMissingMetric(tcf);
        logger.info((Object)"TCF {0} vendor list for version {1}.{2} not found, started downloading.", new Object[]{tcf, this.generationVersion, version});
        this.fetchNewVendorListFor(version);
        return Future.failedFuture((String)"TCF %d vendor list for version %s.%d not fetched yet, try again later.".formatted(tcf, this.generationVersion, version));
    }

    private VendorList toVendorList(String content) {
        try {
            return (VendorList)this.mapper.mapper().readValue(content, VendorList.class);
        }
        catch (IOException e) {
            String message = "Cannot parse vendor list from: " + content;
            logger.error((Object)message, (Throwable)e);
            throw new PreBidException(message, e);
        }
    }

    private Map<Integer, Vendor> filterVendorIdToVendors(VendorList vendorList) {
        return vendorList.getVendors().entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
    }

    private boolean isValid(VendorList vendorList) {
        return vendorList.getVendorListVersion() != null && vendorList.getLastUpdated() != null && MapUtils.isNotEmpty(vendorList.getVendors()) && VendorListService.isValidVendors(vendorList.getVendors().values());
    }

    private static boolean isValidVendors(Collection<Vendor> vendors) {
        return vendors.stream().allMatch(vendor -> vendor != null && vendor.getId() != null && vendor.getPurposes() != null && vendor.getLegIntPurposes() != null && vendor.getFlexiblePurposes() != null && vendor.getSpecialPurposes() != null && vendor.getFeatures() != null && vendor.getSpecialFeatures() != null);
    }

    private int getTcfVersion() {
        return 2;
    }

    private static void createAndCheckWritePermissionsFor(FileSystem fileSystem, String dir) {
        FileProps props;
        FileProps fileProps = props = fileSystem.existsBlocking(dir) ? fileSystem.propsBlocking(dir) : null;
        if (props == null || !props.isDirectory()) {
            try {
                fileSystem.mkdirsBlocking(dir);
            }
            catch (FileSystemException e) {
                throw new PreBidException("Cannot create directory: " + dir, e);
            }
        } else if (!Files.isWritable(Paths.get(dir, new String[0]))) {
            throw new PreBidException("No write permissions for directory: " + dir);
        }
    }

    private Map<Integer, Map<Integer, Vendor>> createCache(FileSystem fileSystem, String cacheDir) {
        Map<String, String> versionToFileContent = this.readFileSystemCache(fileSystem, cacheDir);
        ConcurrentMap cache = Caffeine.newBuilder().build().asMap();
        for (Map.Entry<String, String> versionAndFileContent : versionToFileContent.entrySet()) {
            VendorList vendorList = this.toVendorList(versionAndFileContent.getValue());
            Map<Integer, Vendor> vendorIdToVendors = this.filterVendorIdToVendors(vendorList);
            cache.put(Integer.valueOf(versionAndFileContent.getKey()), vendorIdToVendors);
        }
        return cache;
    }

    private Map<String, String> readFileSystemCache(FileSystem fileSystem, String dir) {
        return fileSystem.readDirBlocking(dir).stream().filter(filepath -> filepath.endsWith(JSON_SUFFIX)).collect(Collectors.toMap(filepath -> StringUtils.removeEnd((String)new File((String)filepath).getName(), (String)JSON_SUFFIX), filename -> fileSystem.readFileBlocking(filename).toString()));
    }

    private Map<Integer, Vendor> readFallbackVendorList(String fallbackVendorListPath) {
        String vendorListContent = this.fileSystem.readFileBlocking(fallbackVendorListPath).toString();
        VendorList vendorList = this.toVendorList(vendorListContent);
        if (!this.isValid(vendorList)) {
            throw new PreBidException("Fallback vendor list parsed but has invalid data: " + vendorListContent);
        }
        return this.filterVendorIdToVendors(vendorList);
    }

    private boolean shouldFallback(int version) {
        return this.deprecated || this.versionsToFallback != null && this.versionsToFallback.contains(version);
    }

    private void fetchNewVendorListFor(int version) {
        String url = this.endpointTemplate.replace(VERSION_PLACEHOLDER, String.valueOf(version));
        this.httpClient.get(url, this.defaultTimeoutMs).map(response -> this.processResponse((HttpClientResponse)response, version)).compose(this::saveToFile).map(this::updateCache).otherwise(exception -> this.handleError((Throwable)exception, version));
    }

    private VendorListResult<VendorList> processResponse(HttpClientResponse response, int version) {
        int statusCode = response.getStatusCode();
        if (statusCode == HttpResponseStatus.NOT_FOUND.code()) {
            throw new MissingVendorListException("Remote server could not found vendor list with version " + this.generationVersion + "." + version);
        }
        if (statusCode != HttpResponseStatus.OK.code()) {
            throw new PreBidException("HTTP status code " + statusCode);
        }
        String body = response.getBody();
        VendorList vendorList = this.toVendorList(body);
        if (!this.isValid(vendorList)) {
            throw new PreBidException("Fetched vendor list parsed but has invalid data: " + body);
        }
        return VendorListResult.of(version, body, vendorList);
    }

    private Future<VendorListResult<VendorList>> saveToFile(VendorListResult<VendorList> vendorListResult) {
        Promise promise = Promise.promise();
        int version = vendorListResult.getVersion();
        String filepath = new File(this.cacheDir, version + JSON_SUFFIX).getPath();
        this.fileSystem.writeFile(filepath, Buffer.buffer((String)vendorListResult.getVendorListAsString()), result -> {
            if (result.succeeded()) {
                promise.complete((Object)vendorListResult);
            } else {
                logger.error((Object)"Could not create new vendor list for version {0}.{1}, file: {2}", result.cause(), new Object[]{this.generationVersion, version, filepath});
                promise.fail(result.cause());
            }
        });
        return promise.future();
    }

    private Void updateCache(VendorListResult<VendorList> vendorListResult) {
        int version = vendorListResult.getVersion();
        this.cache.put(version, this.filterVendorIdToVendors(vendorListResult.getVendorList()));
        int tcf = this.getTcfVersion();
        this.metrics.updatePrivacyTcfVendorListOkMetric(tcf);
        logger.info((Object)"Created new TCF {0} vendor list for version {1}.{2}", new Object[]{tcf, this.generationVersion, version});
        this.stopUsingFallbackForVersion(version);
        return null;
    }

    private Void handleError(Throwable exception, int version) {
        int tcf = this.getTcfVersion();
        this.metrics.updatePrivacyTcfVendorListErrorMetric(tcf);
        if (logger.isDebugEnabled()) {
            logger.debug((Object)"Error while obtaining TCF {0} vendor list for version {1}.{2}", exception, new Object[]{tcf, this.generationVersion, version});
        } else {
            logger.warn((Object)"Error while obtaining TCF {0} vendor list for version {1}.{2}: {3}", new Object[]{tcf, this.generationVersion, version, exception.getMessage()});
        }
        this.startUsingFallbackForVersion(version);
        return null;
    }

    private void startUsingFallbackForVersion(int version) {
        if (this.versionsToFallback == null) {
            return;
        }
        this.versionsToFallback.add(version);
        this.vertx.setTimer(this.refreshMissingListPeriodMs, ignored -> this.fetchNewVendorListFor(version));
    }

    private void stopUsingFallbackForVersion(int version) {
        if (this.versionsToFallback == null) {
            return;
        }
        this.versionsToFallback.remove(version);
    }

    private static class MissingVendorListException
    extends RuntimeException {
        MissingVendorListException(String message) {
            super(message);
        }
    }

    private static final class VendorListResult<T> {
        private final int version;
        private final String vendorListAsString;
        private final T vendorList;

        public int getVersion() {
            return this.version;
        }

        public String getVendorListAsString() {
            return this.vendorListAsString;
        }

        public T getVendorList() {
            return this.vendorList;
        }

        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (!(o instanceof VendorListResult)) {
                return false;
            }
            VendorListResult other = (VendorListResult)o;
            if (this.getVersion() != other.getVersion()) {
                return false;
            }
            String this$vendorListAsString = this.getVendorListAsString();
            String other$vendorListAsString = other.getVendorListAsString();
            if (this$vendorListAsString == null ? other$vendorListAsString != null : !this$vendorListAsString.equals(other$vendorListAsString)) {
                return false;
            }
            T this$vendorList = this.getVendorList();
            T other$vendorList = other.getVendorList();
            return !(this$vendorList == null ? other$vendorList != null : !this$vendorList.equals(other$vendorList));
        }

        public int hashCode() {
            int PRIME = 59;
            int result = 1;
            result = result * 59 + this.getVersion();
            String $vendorListAsString = this.getVendorListAsString();
            result = result * 59 + ($vendorListAsString == null ? 43 : $vendorListAsString.hashCode());
            T $vendorList = this.getVendorList();
            result = result * 59 + ($vendorList == null ? 43 : $vendorList.hashCode());
            return result;
        }

        public String toString() {
            return "VendorListService.VendorListResult(version=" + this.getVersion() + ", vendorListAsString=" + this.getVendorListAsString() + ", vendorList=" + this.getVendorList() + ")";
        }

        @ConstructorProperties(value={"version", "vendorListAsString", "vendorList"})
        private VendorListResult(int version, String vendorListAsString, T vendorList) {
            this.version = version;
            this.vendorListAsString = vendorListAsString;
            this.vendorList = vendorList;
        }

        public static <T> VendorListResult<T> of(int version, String vendorListAsString, T vendorList) {
            return new VendorListResult<T>(version, vendorListAsString, vendorList);
        }
    }
}

