/*
 * Decompiled with CFR 0.152.
 */
package org.prebid.server.validation;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.TreeNode;
import com.fasterxml.jackson.databind.JsonNode;
import com.iab.openrtb.request.Banner;
import com.iab.openrtb.request.BidRequest;
import com.iab.openrtb.request.Deal;
import com.iab.openrtb.request.Format;
import com.iab.openrtb.request.Imp;
import com.iab.openrtb.request.Pmp;
import com.iab.openrtb.request.Site;
import com.iab.openrtb.response.Bid;
import io.vertx.core.logging.Logger;
import io.vertx.core.logging.LoggerFactory;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Currency;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.ListUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.prebid.server.auction.BidderAliases;
import org.prebid.server.auction.model.AuctionContext;
import org.prebid.server.bidder.model.BidderBid;
import org.prebid.server.json.JacksonMapper;
import org.prebid.server.log.ConditionalLogger;
import org.prebid.server.metric.MetricName;
import org.prebid.server.metric.Metrics;
import org.prebid.server.proto.openrtb.ext.request.ExtDeal;
import org.prebid.server.proto.openrtb.ext.request.ExtDealLine;
import org.prebid.server.proto.openrtb.ext.response.BidType;
import org.prebid.server.settings.model.Account;
import org.prebid.server.settings.model.AccountAuctionConfig;
import org.prebid.server.settings.model.AccountBidValidationConfig;
import org.prebid.server.settings.model.BidValidationEnforcement;
import org.prebid.server.validation.ValidationException;
import org.prebid.server.validation.model.ValidationResult;

public class ResponseBidValidator {
    private static final Logger logger = LoggerFactory.getLogger(ResponseBidValidator.class);
    private static final ConditionalLogger UNRELATED_BID_LOGGER = new ConditionalLogger("not_matched_bid", logger);
    private static final ConditionalLogger SECURE_CREATIVE_LOGGER = new ConditionalLogger("secure_creatives_validation", logger);
    private static final ConditionalLogger CREATIVE_SIZE_LOGGER = new ConditionalLogger("creative_size_validation", logger);
    private static final String[] INSECURE_MARKUP_MARKERS = new String[]{"http:", "http%3A"};
    private static final String[] SECURE_MARKUP_MARKERS = new String[]{"https:", "https%3A"};
    private static final String PREBID_EXT = "prebid";
    private static final String BIDDER_EXT = "bidder";
    private static final String DEALS_ONLY = "dealsonly";
    private final BidValidationEnforcement bannerMaxSizeEnforcement;
    private final BidValidationEnforcement secureMarkupEnforcement;
    private final Metrics metrics;
    private final JacksonMapper mapper;
    private final boolean dealsEnabled;
    private final double logSamplingRate;

    public ResponseBidValidator(BidValidationEnforcement bannerMaxSizeEnforcement, BidValidationEnforcement secureMarkupEnforcement, Metrics metrics, JacksonMapper mapper, boolean dealsEnabled, double logSamplingRate) {
        this.bannerMaxSizeEnforcement = Objects.requireNonNull(bannerMaxSizeEnforcement);
        this.secureMarkupEnforcement = Objects.requireNonNull(secureMarkupEnforcement);
        this.metrics = Objects.requireNonNull(metrics);
        this.mapper = Objects.requireNonNull(mapper);
        this.dealsEnabled = dealsEnabled;
        this.logSamplingRate = logSamplingRate;
    }

    public ValidationResult validate(BidderBid bidderBid, String bidder, AuctionContext auctionContext, BidderAliases aliases) {
        Bid bid = bidderBid.getBid();
        BidRequest bidRequest = auctionContext.getBidRequest();
        Account account = auctionContext.getAccount();
        ArrayList<String> warnings = new ArrayList<String>();
        try {
            ResponseBidValidator.validateCommonFields(bid);
            this.validateTypeSpecific(bidderBid, bidder);
            ResponseBidValidator.validateCurrency(bidderBid.getBidCurrency());
            Imp correspondingImp = this.findCorrespondingImp(bid, bidRequest);
            if (bidderBid.getType() == BidType.banner) {
                warnings.addAll(this.validateBannerFields(bid, bidder, bidRequest, account, correspondingImp, aliases));
            }
            if (this.dealsEnabled) {
                this.validateDealsFor(bidderBid, auctionContext.getBidRequest(), bidder, aliases, warnings);
            }
            warnings.addAll(this.validateSecureMarkup(bid, bidder, bidRequest, account, correspondingImp, aliases));
        }
        catch (ValidationException e) {
            return ValidationResult.error(warnings, e.getMessage());
        }
        return ValidationResult.success(warnings);
    }

    private static void validateCommonFields(Bid bid) throws ValidationException {
        if (bid == null) {
            throw new ValidationException("Empty bid object submitted");
        }
        String bidId = bid.getId();
        if (StringUtils.isBlank((CharSequence)bidId)) {
            throw new ValidationException("Bid missing required field 'id'");
        }
        if (StringUtils.isBlank((CharSequence)bid.getImpid())) {
            throw new ValidationException("Bid \"%s\" missing required field 'impid'", bidId);
        }
        if (StringUtils.isEmpty((CharSequence)bid.getCrid())) {
            throw new ValidationException("Bid \"%s\" missing creative ID", bidId);
        }
    }

    private void validateTypeSpecific(BidderBid bidderBid, String bidder) throws ValidationException {
        boolean isVastSpecificAbsent;
        Bid bid = bidderBid.getBid();
        boolean bl = isVastSpecificAbsent = bid.getAdm() == null && bid.getNurl() == null;
        if (Objects.equals((Object)bidderBid.getType(), (Object)BidType.video) && isVastSpecificAbsent) {
            this.metrics.updateAdapterRequestErrorMetric(bidder, MetricName.badserverresponse);
            throw new ValidationException("Bid \"%s\" with video type missing adm and nurl", bid.getId());
        }
    }

    private static void validateCurrency(String currency) throws ValidationException {
        try {
            if (StringUtils.isNotBlank((CharSequence)currency)) {
                Currency.getInstance(currency);
            }
        }
        catch (IllegalArgumentException e) {
            throw new ValidationException("BidResponse currency \"%s\" is not valid", currency);
        }
    }

    private Imp findCorrespondingImp(Bid bid, BidRequest bidRequest) throws ValidationException {
        return bidRequest.getImp().stream().filter(imp -> Objects.equals(imp.getId(), bid.getImpid())).findFirst().orElseThrow(() -> this.exceptionAndLogOnePercent("Bid \"%s\" has no corresponding imp in request".formatted(bid.getId())));
    }

    private ValidationException exceptionAndLogOnePercent(String message) {
        UNRELATED_BID_LOGGER.warn(message, this.logSamplingRate);
        return new ValidationException(message);
    }

    private List<String> validateBannerFields(Bid bid, String bidder, BidRequest bidRequest, Account account, Imp correspondingImp, BidderAliases aliases) throws ValidationException {
        Format maxSize;
        BidValidationEnforcement bannerMaxSizeEnforcement = this.effectiveBannerMaxSizeEnforcement(account);
        if (bannerMaxSizeEnforcement != BidValidationEnforcement.skip && ResponseBidValidator.bannerSizeIsNotValid(bid, maxSize = ResponseBidValidator.maxSizeForBanner(correspondingImp))) {
            String accountId = account.getId();
            String message = "BidResponse validation `%s`: bidder `%s` response triggers creative size validation for bid %s, account=%s, referrer=%s, max imp size='%dx%d', bid response size='%dx%d'".formatted(new Object[]{bannerMaxSizeEnforcement, bidder, bid.getId(), accountId, ResponseBidValidator.getReferer(bidRequest), maxSize.getW(), maxSize.getH(), bid.getW(), bid.getH()});
            return this.singleWarningOrValidationException(bannerMaxSizeEnforcement, metricName -> this.metrics.updateSizeValidationMetrics(aliases.resolveBidder(bidder), accountId, (MetricName)((Object)metricName)), CREATIVE_SIZE_LOGGER, message);
        }
        return Collections.emptyList();
    }

    private BidValidationEnforcement effectiveBannerMaxSizeEnforcement(Account account) {
        AccountAuctionConfig accountAuctionConfig = account.getAuction();
        AccountBidValidationConfig validationConfig = accountAuctionConfig != null ? accountAuctionConfig.getBidValidations() : null;
        BidValidationEnforcement accountBannerMaxSizeEnforcement = validationConfig != null ? validationConfig.getBannerMaxSizeEnforcement() : null;
        return (BidValidationEnforcement)((Object)ObjectUtils.defaultIfNull((Object)((Object)accountBannerMaxSizeEnforcement), (Object)((Object)this.bannerMaxSizeEnforcement)));
    }

    private static Format maxSizeForBanner(Imp imp) {
        int maxW = 0;
        int maxH = 0;
        for (Format size : ResponseBidValidator.bannerFormats(imp)) {
            maxW = Math.max(maxW, size.getW());
            maxH = Math.max(maxH, size.getH());
        }
        return Format.builder().w(maxW).h(maxH).build();
    }

    private static List<Format> bannerFormats(Imp imp) {
        Banner banner = imp.getBanner();
        List<Format> formats = banner != null ? banner.getFormat() : null;
        return ListUtils.emptyIfNull(formats);
    }

    private static boolean bannerSizeIsNotValid(Bid bid, Format maxSize) {
        Integer bidW = bid.getW();
        Integer bidH = bid.getH();
        return bidW == null || bidW > maxSize.getW() || bidH == null || bidH > maxSize.getH();
    }

    private List<String> validateSecureMarkup(Bid bid, String bidder, BidRequest bidRequest, Account account, Imp correspondingImp, BidderAliases aliases) throws ValidationException {
        if (this.secureMarkupEnforcement == BidValidationEnforcement.skip) {
            return Collections.emptyList();
        }
        String accountId = account.getId();
        String referer = ResponseBidValidator.getReferer(bidRequest);
        String adm = bid.getAdm();
        if (ResponseBidValidator.isImpSecure(correspondingImp) && ResponseBidValidator.markupIsNotSecure(adm)) {
            String message = "BidResponse validation `%s`: bidder `%s` response triggers secure creative validation for bid %s, account=%s, referrer=%s, adm=%s".formatted(new Object[]{this.secureMarkupEnforcement, bidder, bid.getId(), accountId, referer, adm});
            return this.singleWarningOrValidationException(this.secureMarkupEnforcement, metricName -> this.metrics.updateSecureValidationMetrics(aliases.resolveBidder(bidder), accountId, (MetricName)((Object)metricName)), SECURE_CREATIVE_LOGGER, message);
        }
        return Collections.emptyList();
    }

    private static boolean isImpSecure(Imp imp) {
        return Objects.equals(imp.getSecure(), 1);
    }

    private static boolean markupIsNotSecure(String adm) {
        return StringUtils.containsAny((CharSequence)adm, (CharSequence[])INSECURE_MARKUP_MARKERS) || !StringUtils.containsAny((CharSequence)adm, (CharSequence[])SECURE_MARKUP_MARKERS);
    }

    private List<String> singleWarningOrValidationException(BidValidationEnforcement enforcement, Consumer<MetricName> metricsRecorder, ConditionalLogger conditionalLogger, String message) throws ValidationException {
        switch (enforcement) {
            default: {
                throw new IncompatibleClassChangeError();
            }
            case enforce: {
                metricsRecorder.accept(MetricName.err);
                conditionalLogger.warn(message, this.logSamplingRate);
                throw new ValidationException(message);
            }
            case warn: {
                metricsRecorder.accept(MetricName.warn);
                conditionalLogger.warn(message, this.logSamplingRate);
                break;
            }
            case skip: {
                throw new IllegalStateException("Unexpected enforcement: " + enforcement);
            }
        }
        return Collections.singletonList(message);
    }

    private static String getReferer(BidRequest bidRequest) {
        Site site = bidRequest.getSite();
        return site != null ? site.getPage() : "unknown";
    }

    private void validateDealsFor(BidderBid bidderBid, BidRequest bidRequest, String bidder, BidderAliases aliases, List<String> warnings) throws ValidationException {
        Bid bid = bidderBid.getBid();
        String bidId = bid.getId();
        Imp imp = bidRequest.getImp().stream().filter(curImp -> Objects.equals(curImp.getId(), bid.getImpid())).findFirst().orElseThrow(() -> new ValidationException("Bid \"%s\" has no corresponding imp in request", bidId));
        String dealId = bid.getDealid();
        if (ResponseBidValidator.isDealsOnlyImp(imp, bidder) && dealId == null) {
            throw new ValidationException("Bid \"%s\" missing required field 'dealid'", bidId);
        }
        if (dealId != null) {
            Set<String> dealIdsFromImp = this.getDealIdsFromImp(imp, bidder, aliases);
            if (CollectionUtils.isNotEmpty(dealIdsFromImp) && !dealIdsFromImp.contains(dealId)) {
                warnings.add("WARNING: Bid \"%s\" has 'dealid' not present in corresponding imp in request. 'dealid' in bid: '%s', deal Ids in imp: '%s'".formatted(bidId, dealId, String.join((CharSequence)",", dealIdsFromImp)));
            }
            if (bidderBid.getType() == BidType.banner) {
                if (imp.getBanner() == null) {
                    throw new ValidationException("Bid \"%s\" has banner media type but corresponding imp in request is missing 'banner' object", bidId);
                }
                List<Format> bannerFormats = ResponseBidValidator.getBannerFormats(imp);
                if (ResponseBidValidator.bidSizeNotInFormats(bid, bannerFormats)) {
                    throw new ValidationException("Bid \"%s\" has 'w' and 'h' not supported by corresponding imp in request. Bid dimensions: '%dx%d', formats in imp: '%s'", bidId, bid.getW(), bid.getH(), ResponseBidValidator.formatSizes(bannerFormats));
                }
                if (this.isPgDeal(imp, dealId)) {
                    this.validateIsInLineItemSizes(bid, bidId, dealId, imp);
                }
            }
        }
    }

    private void validateIsInLineItemSizes(Bid bid, String bidId, String dealId, Imp imp) throws ValidationException {
        List<Format> lineItemSizes = this.getLineItemSizes(imp, dealId);
        if (lineItemSizes.isEmpty()) {
            throw new ValidationException("Line item sizes were not found for bidId %s and dealId %s", bid.getId(), dealId);
        }
        if (ResponseBidValidator.bidSizeNotInFormats(bid, lineItemSizes)) {
            throw new ValidationException("Bid \"%s\" has 'w' and 'h' not matched to Line Item. Bid dimensions: '%dx%d', Line Item sizes: '%s'", bidId, bid.getW(), bid.getH(), ResponseBidValidator.formatSizes(lineItemSizes));
        }
    }

    private static boolean isDealsOnlyImp(Imp imp, String bidder) {
        JsonNode dealsOnlyNode = ResponseBidValidator.bidderParamsFromImp(imp).path(bidder).path(DEALS_ONLY);
        return dealsOnlyNode.isBoolean() && dealsOnlyNode.asBoolean();
    }

    private static JsonNode bidderParamsFromImp(Imp imp) {
        return imp.getExt().path(PREBID_EXT).path(BIDDER_EXT);
    }

    private Set<String> getDealIdsFromImp(Imp imp, String bidder, BidderAliases aliases) {
        return ResponseBidValidator.getDeals(imp).filter(Objects::nonNull).filter(deal -> ResponseBidValidator.isBidderHasDeal(bidder, this.dealExt((JsonNode)deal.getExt()), aliases)).map(Deal::getId).filter(Objects::nonNull).collect(Collectors.toSet());
    }

    private static Stream<Deal> getDeals(Imp imp) {
        Pmp pmp = imp.getPmp();
        return pmp != null ? pmp.getDeals().stream() : Stream.empty();
    }

    private static boolean isBidderHasDeal(String bidder, ExtDeal extDeal, BidderAliases aliases) {
        ExtDealLine extDealLine = extDeal != null ? extDeal.getLine() : null;
        String dealLineBidder = extDealLine != null ? extDealLine.getBidder() : null;
        return dealLineBidder == null || aliases.isSame(bidder, dealLineBidder);
    }

    private static boolean bidSizeNotInFormats(Bid bid, List<Format> formats) {
        return formats.stream().noneMatch(format -> ResponseBidValidator.sizesEqual(bid, format));
    }

    private static boolean sizesEqual(Bid bid, Format format) {
        return Objects.equals(format.getH(), bid.getH()) && Objects.equals(format.getW(), bid.getW());
    }

    private static List<Format> getBannerFormats(Imp imp) {
        return ListUtils.emptyIfNull(imp.getBanner().getFormat());
    }

    private List<Format> getLineItemSizes(Imp imp, String dealId) {
        return ResponseBidValidator.getDeals(imp).filter(deal -> dealId.equals(deal.getId())).map(Deal::getExt).filter(Objects::nonNull).map(this::dealExt).filter(Objects::nonNull).map(ExtDeal::getLine).filter(Objects::nonNull).map(ExtDealLine::getSizes).filter(Objects::nonNull).flatMap(Collection::stream).filter(Objects::nonNull).toList();
    }

    private boolean isPgDeal(Imp imp, String dealId) {
        return ResponseBidValidator.getDeals(imp).filter(Objects::nonNull).filter(deal -> Objects.equals(deal.getId(), dealId)).map(Deal::getExt).filter(Objects::nonNull).map(this::dealExt).filter(Objects::nonNull).map(ExtDeal::getLine).filter(Objects::nonNull).map(ExtDealLine::getLineItemId).anyMatch(Objects::nonNull);
    }

    private ExtDeal dealExt(JsonNode ext) {
        try {
            return (ExtDeal)this.mapper.mapper().treeToValue((TreeNode)ext, ExtDeal.class);
        }
        catch (JsonProcessingException e) {
            logger.warn((Object)"Error decoding deal.ext: {0}", (Throwable)e, new Object[]{e.getMessage()});
            return null;
        }
    }

    private static String formatSizes(List<Format> lineItemSizes) {
        return lineItemSizes.stream().map(ResponseBidValidator::formatSize).collect(Collectors.joining(","));
    }

    private static String formatSize(Format lineItemSize) {
        return "%dx%d".formatted(lineItemSize.getW(), lineItemSize.getH());
    }
}

