/*
 * Decompiled with CFR 0.152.
 */
package org.prebid.server.deals;

import com.fasterxml.jackson.databind.JsonNode;
import com.iab.openrtb.request.BidRequest;
import com.iab.openrtb.request.Imp;
import io.vertx.core.logging.Logger;
import io.vertx.core.logging.LoggerFactory;
import java.math.BigDecimal;
import java.time.Clock;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.ListUtils;
import org.apache.commons.lang3.StringUtils;
import org.prebid.server.auction.BidderAliases;
import org.prebid.server.auction.model.AuctionContext;
import org.prebid.server.currency.CurrencyConversionService;
import org.prebid.server.deals.TargetingService;
import org.prebid.server.deals.events.ApplicationEventService;
import org.prebid.server.deals.lineitem.DeliveryPlan;
import org.prebid.server.deals.lineitem.LineItem;
import org.prebid.server.deals.model.MatchLineItemsResult;
import org.prebid.server.deals.model.TxnLog;
import org.prebid.server.deals.proto.LineItemMetaData;
import org.prebid.server.deals.proto.Price;
import org.prebid.server.deals.targeting.TargetingDefinition;
import org.prebid.server.exception.TargetingSyntaxException;
import org.prebid.server.log.CriteriaLogManager;
import org.prebid.server.proto.openrtb.ext.response.ExtTraceDeal;
import org.prebid.server.util.HttpUtil;

public class LineItemService {
    private static final Logger logger = LoggerFactory.getLogger(LineItemService.class);
    private static final DateTimeFormatter UTC_MILLIS_FORMATTER = new DateTimeFormatterBuilder().appendPattern("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'").toFormatter();
    private static final String ACTIVE = "active";
    private static final String PG_IGNORE_PACING_VALUE = "1";
    private final Comparator<LineItem> lineItemComparator = Comparator.comparing(LineItem::getHighestUnspentTokensClass, Comparator.nullsLast(Comparator.naturalOrder())).thenComparing(LineItem::getRelativePriority, Comparator.nullsLast(Comparator.naturalOrder())).thenComparing(LineItem::getCpm, Comparator.nullsLast(Comparator.reverseOrder()));
    private final int maxDealsPerBidder;
    private final TargetingService targetingService;
    private final CurrencyConversionService conversionService;
    protected final ApplicationEventService applicationEventService;
    private final String adServerCurrency;
    private final Clock clock;
    private final CriteriaLogManager criteriaLogManager;
    protected final Map<String, LineItem> idToLineItems;
    protected volatile boolean isPlannerResponsive;

    public LineItemService(int maxDealsPerBidder, TargetingService targetingService, CurrencyConversionService conversionService, ApplicationEventService applicationEventService, String adServerCurrency, Clock clock, CriteriaLogManager criteriaLogManager) {
        this.maxDealsPerBidder = maxDealsPerBidder;
        this.targetingService = Objects.requireNonNull(targetingService);
        this.conversionService = Objects.requireNonNull(conversionService);
        this.applicationEventService = Objects.requireNonNull(applicationEventService);
        this.adServerCurrency = Objects.requireNonNull(adServerCurrency);
        this.clock = Objects.requireNonNull(clock);
        this.criteriaLogManager = Objects.requireNonNull(criteriaLogManager);
        this.idToLineItems = new ConcurrentHashMap<String, LineItem>();
    }

    public LineItem getLineItemById(String lineItemId) {
        return this.idToLineItems.get(lineItemId);
    }

    public boolean accountHasDeals(AuctionContext auctionContext) {
        return this.accountHasDeals(auctionContext.getAccount().getId(), ZonedDateTime.now(this.clock));
    }

    public boolean accountHasDeals(String account, ZonedDateTime now) {
        return StringUtils.isNotEmpty((CharSequence)account) && this.idToLineItems.values().stream().anyMatch(lineItem -> Objects.equals(lineItem.getAccountId(), account) && lineItem.isActive(now));
    }

    public MatchLineItemsResult findMatchingLineItems(BidRequest bidRequest, Imp imp, String bidder, BidderAliases aliases, AuctionContext auctionContext) {
        ZonedDateTime now = ZonedDateTime.now(this.clock);
        return this.findMatchingLineItems(bidRequest, imp, bidder, aliases, auctionContext, now);
    }

    protected MatchLineItemsResult findMatchingLineItems(BidRequest bidRequest, Imp imp, String bidder, BidderAliases aliases, AuctionContext auctionContext, ZonedDateTime now) {
        List<LineItem> matchedLineItems = this.getPreMatchedLineItems(auctionContext.getAccount().getId(), bidder, aliases).stream().filter(lineItem -> this.isTargetingMatched((LineItem)lineItem, bidRequest, imp, auctionContext)).toList();
        return MatchLineItemsResult.of(this.postProcessMatchedLineItems(matchedLineItems, bidRequest, imp, auctionContext, now));
    }

    public void updateIsPlannerResponsive(boolean isPlannerResponsive) {
        this.isPlannerResponsive = isPlannerResponsive;
    }

    public void updateLineItems(List<LineItemMetaData> planResponse, boolean isPlannerResponsive) {
        this.updateLineItems(planResponse, isPlannerResponsive, ZonedDateTime.now(this.clock));
    }

    public void updateLineItems(List<LineItemMetaData> planResponse, boolean isPlannerResponsive, ZonedDateTime now) {
        this.isPlannerResponsive = isPlannerResponsive;
        if (isPlannerResponsive) {
            List<LineItemMetaData> lineItemsMetaData = ListUtils.emptyIfNull(planResponse).stream().filter(lineItemMetaData -> !this.isExpired(now, lineItemMetaData.getEndTimeStamp())).filter(lineItemMetaData -> Objects.equals(lineItemMetaData.getStatus(), ACTIVE)).toList();
            this.removeInactiveLineItems(planResponse, now);
            lineItemsMetaData.forEach(lineItemMetaData -> this.updateLineItem((LineItemMetaData)lineItemMetaData, now));
        }
    }

    public void invalidateLineItemsByIds(List<String> lineItemIds) {
        this.idToLineItems.entrySet().removeIf(stringLineItemEntry -> lineItemIds.contains(stringLineItemEntry.getKey()));
        logger.info((Object)"Line Items with ids {0} were removed", new Object[]{String.join((CharSequence)", ", lineItemIds)});
    }

    public void invalidateLineItems() {
        String lineItemsToRemove = String.join((CharSequence)", ", this.idToLineItems.keySet());
        this.idToLineItems.clear();
        logger.info((Object)"Line Items with ids {0} were removed", new Object[]{lineItemsToRemove});
    }

    private boolean isExpired(ZonedDateTime now, ZonedDateTime endTime) {
        return now.isAfter(endTime);
    }

    private void removeInactiveLineItems(List<LineItemMetaData> planResponse, ZonedDateTime now) {
        Set lineItemsToRemove = ListUtils.emptyIfNull(planResponse).stream().filter(lineItemMetaData -> !Objects.equals(lineItemMetaData.getStatus(), ACTIVE) || this.isExpired(now, lineItemMetaData.getEndTimeStamp())).map(LineItemMetaData::getLineItemId).collect(Collectors.toSet());
        this.idToLineItems.entrySet().stream().filter(entry -> this.isExpired(now, ((LineItem)entry.getValue()).getEndTimeStamp())).map(Map.Entry::getKey).collect(Collectors.toCollection(() -> lineItemsToRemove));
        if (CollectionUtils.isNotEmpty(lineItemsToRemove)) {
            logger.info((Object)"Line Items {0} were dropped as expired or inactive", new Object[]{String.join((CharSequence)", ", lineItemsToRemove)});
        }
        this.idToLineItems.entrySet().removeIf(entry -> lineItemsToRemove.contains(entry.getKey()));
    }

    protected Collection<LineItem> getLineItems() {
        return this.idToLineItems.values();
    }

    protected void updateLineItem(LineItemMetaData lineItemMetaData, ZonedDateTime now) {
        TargetingDefinition targetingDefinition = this.makeTargeting(lineItemMetaData);
        Price normalizedPrice = this.normalizedPrice(lineItemMetaData);
        this.idToLineItems.compute(lineItemMetaData.getLineItemId(), (id, li) -> li != null ? li.withUpdatedMetadata(lineItemMetaData, normalizedPrice, targetingDefinition, li.getReadyAt(), now) : LineItem.of(lineItemMetaData, normalizedPrice, targetingDefinition, now));
    }

    public void advanceToNextPlan(ZonedDateTime now) {
        Collection<LineItem> lineItems = this.idToLineItems.values();
        for (LineItem lineItem : lineItems) {
            lineItem.advanceToNextPlan(now, this.isPlannerResponsive);
        }
        this.applicationEventService.publishDeliveryUpdateEvent();
    }

    private TargetingDefinition makeTargeting(LineItemMetaData lineItemMetaData) {
        TargetingDefinition targetingDefinition;
        try {
            targetingDefinition = this.targetingService.parseTargetingDefinition((JsonNode)lineItemMetaData.getTargeting(), lineItemMetaData.getLineItemId());
        }
        catch (TargetingSyntaxException e) {
            this.criteriaLogManager.log(logger, lineItemMetaData.getAccountId(), lineItemMetaData.getSource(), lineItemMetaData.getLineItemId(), "Line item targeting parsing failed with a reason: " + e.getMessage(), arg_0 -> ((Logger)logger).warn(arg_0));
            targetingDefinition = null;
        }
        return targetingDefinition;
    }

    private Price normalizedPrice(LineItemMetaData lineItemMetaData) {
        Price price = lineItemMetaData.getPrice();
        if (price == null) {
            return null;
        }
        String receivedCur = price.getCurrency();
        if (StringUtils.equals((CharSequence)this.adServerCurrency, (CharSequence)receivedCur)) {
            return price;
        }
        BigDecimal updatedCpm = this.conversionService.convertCurrency(price.getCpm(), Collections.emptyMap(), receivedCur, this.adServerCurrency, null);
        return Price.of(updatedCpm, this.adServerCurrency);
    }

    private List<LineItem> getPreMatchedLineItems(String accountId, String bidder, BidderAliases aliases) {
        if (StringUtils.isBlank((CharSequence)accountId)) {
            return Collections.emptyList();
        }
        List<LineItem> accountsLineItems = this.idToLineItems.values().stream().filter(lineItem -> lineItem.getAccountId().equals(accountId)).toList();
        if (accountsLineItems.isEmpty()) {
            this.criteriaLogManager.log(logger, accountId, (Object)("There are no line items for account " + accountId), arg_0 -> ((Logger)logger).debug(arg_0));
            return Collections.emptyList();
        }
        return accountsLineItems.stream().filter(lineItem -> aliases.isSame(bidder, lineItem.getSource())).toList();
    }

    private boolean isTargetingMatched(LineItem lineItem, BidRequest bidRequest, Imp imp, AuctionContext auctionContext) {
        TargetingDefinition targetingDefinition = lineItem.getTargetingDefinition();
        String accountId = auctionContext.getAccount().getId();
        String source = lineItem.getSource();
        String lineItemId = lineItem.getLineItemId();
        if (targetingDefinition == null) {
            this.deepDebug(auctionContext, ExtTraceDeal.Category.targeting, "Line Item %s targeting was not defined or has incorrect format".formatted(lineItemId), accountId, source, lineItemId);
            return false;
        }
        boolean matched = this.targetingService.matchesTargeting(bidRequest, imp, lineItem.getTargetingDefinition(), auctionContext);
        String debugMessage = matched ? "Line Item %s targeting matched imp with id %s".formatted(lineItemId, imp.getId()) : "Line Item %s targeting did not match imp with id %s".formatted(lineItemId, imp.getId());
        this.deepDebug(auctionContext, ExtTraceDeal.Category.targeting, debugMessage, accountId, source, lineItemId);
        return matched;
    }

    private List<LineItem> postProcessMatchedLineItems(List<LineItem> lineItems, BidRequest bidRequest, Imp imp, AuctionContext auctionContext, ZonedDateTime now) {
        TxnLog txnLog = auctionContext.getTxnLog();
        List<String> fcapIds = bidRequest.getUser().getExt().getFcapIds();
        return lineItems.stream().peek(lineItem -> txnLog.lineItemsMatchedWholeTargeting().add(lineItem.getLineItemId())).filter(lineItem -> this.isNotFrequencyCapped(fcapIds, (LineItem)lineItem, auctionContext, txnLog)).filter(lineItem -> this.planHasTokensIfPresent((LineItem)lineItem, auctionContext)).filter(lineItem -> this.isReadyAtInPast(now, (LineItem)lineItem, auctionContext, txnLog)).peek(lineItem -> txnLog.lineItemsReadyToServe().add(lineItem.getLineItemId())).collect(Collectors.groupingBy(lineItem -> lineItem.getSource().toLowerCase())).values().stream().map(valueAsLineItems -> this.filterLineItemPerBidder((List<LineItem>)valueAsLineItems, auctionContext, imp)).filter(CollectionUtils::isNotEmpty).peek(lineItemsForBidder -> LineItemService.recordInTxnSentToBidderAsTopMatch(txnLog, lineItemsForBidder)).flatMap(Collection::stream).peek(lineItem -> txnLog.lineItemsSentToBidder().get(lineItem.getSource()).add(lineItem.getLineItemId())).toList();
    }

    private boolean planHasTokensIfPresent(LineItem lineItem, AuctionContext auctionContext) {
        if (this.hasUnspentTokens(lineItem) || LineItemService.ignorePacing(auctionContext)) {
            return true;
        }
        String lineItemId = lineItem.getLineItemId();
        String lineItemSource = lineItem.getSource();
        auctionContext.getTxnLog().lineItemsPacingDeferred().add(lineItemId);
        this.deepDebug(auctionContext, ExtTraceDeal.Category.pacing, "Matched Line Item %s for bidder %s does not have unspent tokens to be served".formatted(lineItemId, lineItemSource), auctionContext.getAccount().getId(), lineItemSource, lineItemId);
        return false;
    }

    private boolean hasUnspentTokens(LineItem lineItem) {
        DeliveryPlan deliveryPlan = lineItem.getActiveDeliveryPlan();
        return deliveryPlan == null || deliveryPlan.getDeliveryTokens().stream().anyMatch(deliveryToken -> deliveryToken.getUnspent() > 0);
    }

    private static boolean ignorePacing(AuctionContext auctionContext) {
        return PG_IGNORE_PACING_VALUE.equals(auctionContext.getHttpRequest().getHeaders().get(HttpUtil.PG_IGNORE_PACING));
    }

    private boolean isReadyAtInPast(ZonedDateTime now, LineItem lineItem, AuctionContext auctionContext, TxnLog txnLog) {
        ZonedDateTime readyAt = lineItem.getReadyAt();
        boolean ready = readyAt != null && LineItemService.isBeforeOrEqual(readyAt, now) || LineItemService.ignorePacing(auctionContext);
        String accountId = auctionContext.getAccount().getId();
        String lineItemSource = lineItem.getSource();
        String lineItemId = lineItem.getLineItemId();
        if (ready) {
            this.deepDebug(auctionContext, ExtTraceDeal.Category.pacing, "Matched Line Item %s for bidder %s ready to serve. relPriority %d".formatted(lineItemId, lineItemSource, lineItem.getRelativePriority()), accountId, lineItemSource, lineItemId);
        } else {
            txnLog.lineItemsPacingDeferred().add(lineItemId);
            this.deepDebug(auctionContext, ExtTraceDeal.Category.pacing, "Matched Line Item %s for bidder %s not ready to serve. Will be ready at %s, current time is %s".formatted(lineItemId, lineItemSource, readyAt != null ? UTC_MILLIS_FORMATTER.format(readyAt) : "never", UTC_MILLIS_FORMATTER.format(now)), accountId, lineItemSource, lineItemId);
        }
        return ready;
    }

    private static boolean isBeforeOrEqual(ZonedDateTime before, ZonedDateTime after) {
        return before.isBefore(after) || before.isEqual(after);
    }

    private boolean isNotFrequencyCapped(List<String> frequencyCappedByIds, LineItem lineItem, AuctionContext auctionContext, TxnLog txnLog) {
        if (CollectionUtils.isEmpty(lineItem.getFcapIds())) {
            return true;
        }
        String lineItemId = lineItem.getLineItemId();
        String accountId = auctionContext.getAccount().getId();
        String lineItemSource = lineItem.getSource();
        if (frequencyCappedByIds == null) {
            txnLog.lineItemsMatchedTargetingFcapLookupFailed().add(lineItemId);
            String message = "Failed to match fcap for Line Item %s bidder %s in a reason of bad response from user data service".formatted(lineItemId, lineItemSource);
            this.deepDebug(auctionContext, ExtTraceDeal.Category.pacing, message, accountId, lineItemSource, lineItemId);
            this.criteriaLogManager.log(logger, lineItem.getAccountId(), lineItem.getSource(), lineItemId, "Failed to match fcap for lineItem %s in a reason of bad response from user data service".formatted(lineItemId), arg_0 -> ((Logger)logger).debug(arg_0));
            return false;
        }
        if (!frequencyCappedByIds.isEmpty()) {
            Optional<String> fcapIdOptional = lineItem.getFcapIds().stream().filter(frequencyCappedByIds::contains).findFirst();
            if (fcapIdOptional.isPresent()) {
                String fcapId = fcapIdOptional.get();
                txnLog.lineItemsMatchedTargetingFcapped().add(lineItemId);
                String message = "Matched Line Item %s for bidder %s is frequency capped by fcap id %s.".formatted(lineItemId, lineItemSource, fcapId);
                this.deepDebug(auctionContext, ExtTraceDeal.Category.pacing, message, accountId, lineItemSource, lineItemId);
                this.criteriaLogManager.log(logger, lineItem.getAccountId(), lineItem.getSource(), lineItemId, message, arg_0 -> ((Logger)logger).debug(arg_0));
                return false;
            }
        }
        return true;
    }

    private List<LineItem> filterLineItemPerBidder(List<LineItem> lineItems, AuctionContext auctionContext, Imp imp) {
        ArrayList<LineItem> sortedLineItems = new ArrayList<LineItem>(lineItems);
        Collections.shuffle(sortedLineItems);
        sortedLineItems.sort(this.lineItemComparator);
        List<LineItem> filteredLineItems = this.uniqueBySentToBidderAsTopMatch(sortedLineItems, auctionContext, imp);
        this.updateLostToLineItems(filteredLineItems, auctionContext.getTxnLog());
        HashSet<String> dealIds = new HashSet<String>();
        ArrayList<LineItem> resolvedLineItems = new ArrayList<LineItem>();
        for (LineItem lineItem : filteredLineItems) {
            String dealId = lineItem.getDealId();
            if (dealIds.contains(dealId)) continue;
            dealIds.add(dealId);
            resolvedLineItems.add(lineItem);
        }
        return resolvedLineItems.size() > this.maxDealsPerBidder ? this.cutLineItemsToDealMaxNumber(resolvedLineItems) : resolvedLineItems;
    }

    private List<LineItem> uniqueBySentToBidderAsTopMatch(List<LineItem> lineItems, AuctionContext auctionContext, Imp imp) {
        TxnLog txnLog = auctionContext.getTxnLog();
        Set topMatchedLineItems = txnLog.lineItemsSentToBidderAsTopMatch().values().stream().flatMap(Collection::stream).collect(Collectors.toSet());
        ArrayList<LineItem> result = new ArrayList<LineItem>(lineItems);
        for (LineItem lineItem : lineItems) {
            String lineItemId = lineItem.getLineItemId();
            if (!topMatchedLineItems.contains(lineItemId)) {
                return result;
            }
            result.remove(lineItem);
            this.deepDebug(auctionContext, ExtTraceDeal.Category.cleanup, "LineItem %s was dropped from imp with id %s because it was top match in another imp".formatted(lineItemId, imp.getId()), auctionContext.getAccount().getId(), lineItem.getSource(), lineItemId);
        }
        return result;
    }

    private List<LineItem> cutLineItemsToDealMaxNumber(List<LineItem> resolvedLineItems) {
        resolvedLineItems.subList(this.maxDealsPerBidder, resolvedLineItems.size()).forEach(lineItem -> this.criteriaLogManager.log(logger, lineItem.getAccountId(), lineItem.getSource(), lineItem.getLineItemId(), "LineItem %s was dropped by max deal per bidder limit %s".formatted(lineItem.getLineItemId(), this.maxDealsPerBidder), arg_0 -> ((Logger)logger).debug(arg_0)));
        return resolvedLineItems.subList(0, this.maxDealsPerBidder);
    }

    private void updateLostToLineItems(List<LineItem> lineItems, TxnLog txnLog) {
        for (int i = 1; i < lineItems.size(); ++i) {
            LineItem lineItem = lineItems.get(i);
            Set lostTo = lineItems.subList(0, i).stream().map(LineItem::getLineItemId).collect(Collectors.toSet());
            txnLog.lostMatchingToLineItems().put(lineItem.getLineItemId(), lostTo);
        }
    }

    private void deepDebug(AuctionContext auctionContext, ExtTraceDeal.Category category, String message, String accountId, String bidder, String lineItemId) {
        this.criteriaLogManager.log(logger, accountId, bidder, lineItemId, message, arg_0 -> ((Logger)logger).debug(arg_0));
        auctionContext.getDeepDebugLog().add(lineItemId, category, () -> message);
    }

    private static void recordInTxnSentToBidderAsTopMatch(TxnLog txnLog, List<LineItem> lineItemsForBidder) {
        LineItem topLineItem = lineItemsForBidder.get(0);
        txnLog.lineItemsSentToBidderAsTopMatch().get(topLineItem.getSource()).add(topLineItem.getLineItemId());
    }
}

